'Pretty Garage 2024-01-19
'Added repository support 2024-02-03
'Added car images 2024-02-09
'Added support for online groups 2024-03-07
'Changed car config file extensions from .cfg to .ccf 2025-01-05
'Working on setting RAMdir and ROMdir 2025-01-06
'Added detection of Stunts directory 2025-01-17
'Added current panel marker 2025-01-17
'Added filtering commands 2025-01-17
'Added setauthor command 2025-01-17

Declare Function GetCarIcon Lib "renderer" (shapefile As String, paintjob As Byte) As Any Ptr
Declare Sub TargaSave Lib "renderer" (filename As String, image As Any Ptr)
Declare Function TargaLoad Lib "renderer" (filename As String, pal As UByte Ptr = 0) As Any Pointer
Declare Sub SortList (which As Byte, by As Byte)
Declare Sub UpdateRepo (id As Byte = 0)

#include "file.bi"

'Maximum cars per panel
#define MAX_CARS 300

#ifdef __FB_LINUX__
	#define SEP "/"
	#define ALLFILES "*"
#else
	#define SEP "\"
	#define ALLFILES "*.*"
#endif

Type Car
	id As String * 4
	name As String
	author As String
	where As Short		'0 = Stunts, + = Garage, - = Respository
	infofile As String
	paramfile As String
	shapefile As String
	imagefile As String
	cfile(1 To 8) As String
	cfiles As Byte
	onlineloc As String
	selected As Byte
End Type

Type Panel
	name As String	'Panel name/title
	type As Byte	'0 = Stunts, 1 = Garage, -1 = Repository, 2 = Group
	id As Short		'Number of whatever it is
	n As Short		'Number of cars
	first As Short	'First car shown
End Type


Dim Shared car(0 To 1, 1 To MAX_CARS) As Car, cars(0 To 1) As Short
Dim Shared pan(0 To 1) As Panel, sdir As String
Dim Shared gdir(1 To 10) As String, gdirs As Byte, gdirname(1 To 10) As String
Dim Shared repo(1 To 10) As String, repos As Byte, reponame(1 To 10) As String
Dim Shared group(1 To 20) As String, groups As Byte, groupname(1 To 20) As String
Dim Shared swidth As Short = 1024, sheight As Short = 768
Dim Shared As String unzipcom, autoupdate, lastupdate, stunpack

Dim Shared As Integer xm, ym, wm, bm, exxm, exym, exwm, exbm
Dim Shared As Byte zone = -1, exzone = -1, sortmethod = 0, showauthors = -1
Dim Shared panelmessage As String, defaultcar As String, curpan As Byte, action As Short

Dim Shared As String ROMdir, RAMdir


Function ISODate As String
	Return Right(Date, 4) & "-" & Left(Date, 2) & "-" & Mid(Date, 4, 2)
End Function


Function FindStunts (where As String) As String
	Dim localfile(1 To 200) As String, localfiles As UByte
	Dim s As String, searchdir As String, q As Byte
	
	searchdir = where
	If Right(searchdir, 1) <> SEP Then searchdir &= SEP
	
	'First, see if Stunts is here
	#ifdef __FB_LINUX__
		s = Dir(searchdir & "*", 0)
	#else
		s = Dir(searchdir & "*.*", 0)
	#endif
	q = 0
	Do
		If Len(s) = 0 Then Exit Do
		If LCase(s) = "opp3.pre" OrElse LCase(s) = "game2.p3s" Then
			q += 1
			'Found Stunts here
			If q = 2 Then Return where
		End If
		
		s = Dir
	Loop
	
	'Now, check the subdirectories
	#ifdef __FB_LINUX__
		s = Dir(searchdir & "*", 16)
	#else
		s = Dir(searchdir & "*.*", 16)
	#endif
	Do
		If Len(s) Then
			If s <> "." AndAlso s <> ".." Then
				localfiles += 1
				localfile(localfiles) = s
			End If
		Else
			Exit Do
		End If
		s = Dir
	Loop Until localfiles = 200
	
	If localfiles = 0 Then
		'Directory structure ends here, got to leaf
		Return ""
	End If
	
	For i As UByte = 1 To localfiles
		s = FindStunts(searchdir & localfile(i))
		'Found Stunts in a subdirectory
		If Len(s) Then Return s
	Next i
	
	'Stunts wasn't found in any subdirectory
	Return ""
End Function


Function FindStuntsByOS As String
	Dim rootdir(1 To 101) As String, rootdirs As UByte
	Dim s As String
	
	#ifdef __FB_LINUX__
		s = Dir("/*", 16)
		Do
			If Len(s) Then
				If InStr("/../bin/proc/root/boot/dev/cdrom/home/media/run/srv/sys/", s) = 0 Then
					rootdirs += 1
					rootdir(rootdirs) = "/" & s
				End If
			Else
				Exit Do
			End If
			s = Dir
		Loop Until rootdirs = 100
		rootdirs += 1
		rootdir(rootdirs) = Environ("HOME")
	#else
		rootdirs = 3
		rootdir(1) = "C:"
		rootdir(2) = "D:"
		rootdir(3) = "E:"
	#endif
	
	For i As UByte = 1 To rootdirs
		s = FindStunts(rootdir(i))
		If Len(s) Then Return s
	Next i
	
	Return ""
End Function


Sub Initialise
	'Identify ROMdir, that is, where static data files will be found
	If FileExists("defmat.txt") Then
		ROMdir = "." & SEP
	ElseIf FileExists(ExePath & SEP & "defmat.txt") Then
		ROMdir = ExePath & SEP
	Else
		Screen 13
		Width 40, 25
		Color 12
		Print "Error!"
		Color 7
		Print "No data files found in current or"
		Print "program binary directory. Press any key"
		Print
		GetKey
		Screen 0
		End 1
	End If
	
	'Identify RAMdir, that is, where the configuration and other
	'changing files will be stored
	Dim As Byte cfg(1 To 3), wri(1 To 3)
	
	If FileExists("pgarage.cfg") Then cfg(1) = -1
	If FileExists(ExePath & SEP & "pgarage.cfg") Then cfg(2) = -1
	If FileExists(Environ("HOME") & SEP & ".pgarage/pgarage.cfg") Then cfg(3) = -1
	
	Dim f As Integer
	f = FreeFile
	Open "xrutest.bin" For Output As f
	If Err = 0 Then Close f : Kill "xrutest.bin" : wri(1) = -1
	Open ExePath & SEP & "xrutest.bin" For Output As f
	If Err = 0 Then Close f : Kill ExePath & SEP & "xrutest.bin" : wri(2) = -1
	Open Environ("HOME") & SEP & ".pgarage/xrutest.bin" For Output As f
	If Err = 0 Then Close f : Kill Environ("HOME") & SEP & ".pgarage/xrutest.bin" : wri(3) = -1

	If cfg(1) AndAlso wri(1) Then
		RAMdir = "." & SEP
	ElseIf cfg(2) AndAlso wri(2) Then
		RAMdir = ExePath & SEP
	ElseIf FileExists(Environ("HOME")) Then
		If Not FileExists(Environ("HOME") & SEP & ".pgarage") Then MkDir Environ("HOME") & SEP & ".pgarage"
		If cfg(2) AndAlso Not cfg(3) Then FileCopy ExePath & SEP & "pgarage.cfg", Environ("HOME") & SEP & ".pgarage" & SEP & "pgarage.cfg"
		RAMdir = Environ("HOME") & SEP & ".pgarage" & SEP
	ElseIf cfg(2) AndAlso wri(1) Then
		FileCopy ExePath & SEP & "pgarage.cfg", "." & SEP & "pgarage.cfg"
		RAMdir = "." & SEP
	Else
		Screen 13
		Width 40, 25
		Color 12
		Print "Error!"
		Color 7
		Print "Current directory not writable and"
		Print "could not locate pgarage.cfg. Press a key"
		Print
		GetKey
		Screen 0
		End 1
	End If
End Sub


Sub LoadConfig
	If Not FileExists(RAMdir & "pgarage.cfg") Then Exit Sub
	
	Dim f As Integer, s As String, n As Short, k As String, v As String
	
	defaultcar = "coun"
	autoupdate = "yes"
	f = FreeFile
	Open RAMdir & "pgarage.cfg" For Input As f
	While Not EoF(f)
		Line Input #f, s
		n = InStr(s, "=")
		
		If n <> 0 And Left(LTrim(s), 1) <> "#" Then
			k = LCase(Trim(Left(s, n - 1)))
			v = Trim(Mid(s, n + 1))
			
			Select Case k
				Case "stunts", "sdir", "stuntsdir" : sdir = v
				Case "garage", "gdir", "gar"
					If gdirs < UBound(gdir) Then
						gdirs += 1
						n = InStr(v, "|")
						If n Then
							gdir(gdirs) = Trim(Mid(v, n + 1))
							gdirname(gdirs)= Trim(Left(v, n - 1))
						Else
							gdir(gdirs) = v
							n = InStrRev(v, SEP)
							If n Then
								gdirname(gdirs) = Mid(v, n + 1)
							Else
								gdirname(gdirs) = v
							End If
						End If
					End If
				Case "group"
					If groups < UBound(group) Then
						groups += 1
						n = InStr(v, "|")
						If n Then
							group(groups) = Trim(Mid(v, n + 1))
							groupname(groups) = Trim(Left(v, n - 1))
						Else
							group(groups) = v
							groupname(groups) = "Group #" & groups
						End If
					End If
				Case "repo", "repository"
					If repos < UBound(repo) Then
						repos += 1
						n = InStr(v, "|")
						If n Then
							repo(repos) = Trim(Mid(v, n + 1))
							reponame(repos) = Trim(Left(v, n - 1))
						Else
							repo(repos) = v
							n = InStrRev(v, ".")
							If n Then v = Left(v, n - 1)
							n = InStrRev(v, "/")
							If n Then v = Mid(v, n + 1)
							n = InStrRev(v, ":")
							If n Then v = Mid(v, n + 1)
							reponame(repos) = v
						End If
					End If
				Case "width" : swidth = ValInt(v)
				Case "height" : sheight = ValInt(v)
				Case "defaultcar" : defaultcar = LCase(Trim(v))
				Case "unzip" : unzipcom = Trim(v)
				Case "lastupdate" : lastupdate = Trim(v)
				Case "autoupdate" : autoupdate = LCase(Trim(v))
				Case "stunpack" : stunpack = Trim(v)
			End Select
		End If
	WEnd
	Close f
End Sub


Sub SaveCarConfig(c As Car)
	Dim path As String, f As Integer, n As Short
	Dim li(1 To 100) As String, lis As Short, q As Byte
	
	'Only possible for cars in Stunts or a garage
	If c.where < 0 Then Exit Sub
	
	f = FreeFile
	
	If c.where = 0 Then
		path = sdir
	Else
		path = gdir(c.where)
	End If
	
	If Len(c.infofile) = 0 Then
		If FileExists(path & SEP & "CAR" & UCase(c.id) & ".CCF") Then
			c.infofile = "CAR" & UCase(c.id) & ".CCF"
		Else
			c.infofile = "car" & c.id & ".ccf"
		End If
	End If
	
	If FileExists(path & SEP & c.infofile) Then
		f = FreeFile
		Open path & SEP & c.infofile For Input As f
		While Not EoF(f)
			lis += 1
			Line Input #f, li(lis)
			If lis = 99 Then Exit While
		WEnd
		Close f
	End If
	
	'As of now, the only car configuration we can change and save
	'is the car's author
	For i As UByte = 1 To lis
		n = InStr(li(i), "=")
		If n = 0 Then n = InStr(li(i), ":")
		If n Then
			If LCase(Trim(Left(li(i), n - 1))) = "author" Then
				li(i) = Left(li(i), n) + c.author
				q = -1
			End If
		End If
	Next i
	If q = 0 Then lis += 1 : li(lis) = "author=" + c.author
	
	Open path & SEP & c.infofile For Output As f
	For i As UByte = 1 To lis
		Print #f, li(i)
	Next i
	Close f
End Sub


Sub DetectGoodDirs
	'Autodetect if not there
	If Len(sdir) = 0 Then
		Print
		Print "Finding Stunts..."
		sdir = FindStuntsByOS
		If Len(sdir) Then Print "Found at " & sdir Else Print "Could not find Stunts"
	End If

	'See if Stunts directory can be written to if found
	Dim As Byte stwrite = -1
	If Len(sdir) Then
		Dim f As Integer
		f = FreeFile
		Open sdir & SEP & "xrutest.bin" For Output As f
		If Err Then
			'Stunts directory not writable
			stwrite = 0
		Else
			Close f : Kill sdir & SEP & "xrutest.bin"
		End If
	End If
	
	'In that case, if no garage, try to create one there
	If gdirs = 0 AndAlso Len(sdir) <> 0 Then
		If Dir(sdir & SEP & "garage", 16) = "garage" Then
			'There seems to already be a garage directory here
			gdirs = 1 : gdir(1) = sdir & SEP & "garage"
		Else
			'Try to create a garage there
			MkDir sdir & SEP & "garage"
			If Err = 0 Then
				gdirs = 1 : gdir(1) = sdir & SEP & "garage"
			End If
		End If
	End If
	
	'If nothing else, have the user specify the directories
	'Finally check that the garage can be written to too

	'Check that garage is OK
	Dim garwrite As Byte = -1
	If gdirs Then
		Dim f As Integer
		f = FreeFile
		Open gdir(1) & SEP & "xrutest.bin" For Output As f
		If Err Then
			'Main garage directory not writable
			garwrite = 0
		Else
			Close f : Kill gdir(1) & SEP & "xrutest.bin"
		End If
	End If
	
	If Len(sdir) = 0 OrElse gdirs = 0 OrElse Len(gdir(1)) = 0 OrElse stwrite = 0 OrElse garwrite = 0 Then
		Screen 12
		Width 80, 30
		
		If Len(sdir) = 0 Then
			Print "No Stunts directory specified in configuration"
			Print "Could not find where Stunts is located"
		ElseIf stwrite = 0 Then
			Print "Found Stunts at " & sdir & " but cannot write there"
			Print "so moving cars won't be possible"
		End If
		
		If gdirs = 0 OrElse Len(gdir(1)) = 0 Then
			Print "No garage directory specified or detected"
		ElseIf garwrite = 0 Then
			Print "Garage at " & gdir(1) & " cannot be written to"
		End If
		
		Print
		Print "Please edit the pgarage.cfg file to properly set the directories"
		Print "and ensure they are accessible and writable. Press any key"
		GetKey
		End 1
	End If
End Sub


Function CarImage(c As Car, force As Byte = 0, paintjob As Byte = 1) As Any Ptr
	Dim sourcedir As String, myimg As Any Ptr
	
	'If there's no cachéd image, try to generate one
	If force OrElse Not FileExists(ROMdir & "icons" & SEP & LCase(c.id) & ".tga") Then
		If c.where = 0 Then
			'It's in Stunts
			sourcedir = sdir
		ElseIf c.where > 0 Then
			'It's in a garage
			sourcedir = gdir(c.where)
		Else
			'Nothing to do here. We have no image!
			Return 0
		End If
		
		If FileExists(sourcedir & SEP & c.shapefile) Then
			If LCase(Right(c.shapefile, 4)) = ".3sh" Then
				'Found a shape from which to generate the image
				myimg = GetCarIcon(sourcedir & SEP & c.shapefile, paintjob)
				If myimg Then Kill "icons" & SEP & LCase(c.id) & ".tga"
				TargaSave "icons" & SEP & LCase(c.id) & ".tga", myimg
				Return myimg
			ElseIf LCase(Right(c.shapefile, 4)) = ".p3s" Then
				'Found a shape, but it's packed. See if we have stunpack
				If Len(stunpack) Then
					Open Pipe stunpack & " " & sourcedir & SEP & c.shapefile & " ." & SEP & "temp.3sh" For Input As 97
					Dim s As String
					While Not EoF(97)
						Line Input #97, s
					WEnd
					Close 97
					If FileExists("." & SEP & "temp.3sh") Then
						myimg = GetCarIcon("." & SEP & "temp.3sh", paintjob)
						TargaSave "icons" & SEP & LCase(c.id) & ".tga", myimg
						Kill "." & SEP & "temp.3sh"
						Return myimg
					Else
						Return 0
					End If
				End If
			End If
			'Could not retrieve an image
			Return 0
		Else
			'No shape file. Nothing to do either!
			Return 0
		End If
	Else
		myimg = TargaLoad(ROMdir & "icons" & SEP & LCase(c.id) & ".tga")
		Return myimg
	End If
End Function


Sub DrawPanel(n As Byte, param As Short = -1)
	Dim As Short x1, y1, x2, y2
	
	Select Case n
		Case 0
			x1 = 0 : x2 = (swidth - 200) \ 2 - 8
			y1 = 0 : y2 = sheight - 108
		Case 1
			x1 = (swidth - 200) \ 2 : x2 = swidth - 208
			y1 = 0 : y2 = sheight - 108
		Case 2
			x1 = swidth - 200 : x2 = swidth - 1
			y1 = 0 : y2 = sheight - 108
		Case 3
			x1 = 0 : x2 = swidth - 1
			y1 = sheight - 100 : y2 = sheight -1
	End Select
	
	ScreenLock
	Line (x1, y1)-(x2, y2), RGB(20, 20, 20), BF
	Line (x1, y2 - 1)-(x1, y1), RGB(40, 40, 40)
	Line - (x2 - 1, y1), RGB(40, 40, 40)
	Line (x2, y1 + 1)-(x2, y2), RGB(10, 10, 10)
	Line -(x1 + 1, y2), RGB(10, 10, 10)
	
	Dim yline As Short
	
	Select Case n
		Case 0, 1
			Dim pancol As ULong
			
			If pan(n).type = 0 Then
				pancol = RGB(250, 250, 0)
				Draw String (x1 + 16, y1 + 8), "Stunts", pancol
			ElseIf pan(n).type = 1 Then
				pancol = RGB(0, 250, 0)
				Draw String (x1 + 16, y1 + 8), pan(n).name, pancol
			ElseIf pan(n).type = 2 Then
				pancol = RGB(250, 100, 0)
				Draw String (x1 + 16, y1 + 8), pan(n).name, pancol
			ElseIf pan(n).type = -1 Then
				pancol = RGB(0, 250, 250)
				Draw String (x1 + 16, y1 + 8), pan(n).name, pancol
			End If
			
			If curpan = n Then
				Circle (x1 + 8, y1 + 14), 3, pancol, , , , F
			End If
			
			Dim sels As Short, s As String
			
			For i As Short = 1 To cars(n)
				If car(n, i).selected Then sels += 1
			Next i
			
			s = cars(n) & IIf(cars(n) = 1, " car", " cars")
			If sels Then s &= ", " & sels & " selected"
			
			Draw String (x1 + 16, y1 + 24), s, RGB(120, 120, 120)
			Line (x1 + 8, y1 + 39)-(x2 - 8, y1 + 39), RGB(10, 10, 10)
			Line (x1 + 8, y1 + 40)-(x2 - 8, y1 + 40), RGB(40, 40, 40)
			
			Dim As Short yoff, panheight
			
			panheight = (y2 - 4) - (y1 + 48)
			If panheight \ 40 < cars(n) AndAlso pan(n).first + panheight \ 40 > cars(n) + 1 Then pan(n).first = cars(n) + 1 - panheight \ 40
						
			View Screen (x1 + 4, y1 + 4)-(x2 - 4, y2 - 4)
			yoff = y1 + 48
			For i As Short = pan(n).first To cars(n)
				Dim s As String, l As Short, nextx As Short
				
				If car(n, i).selected Then Line (x1 + 4, yoff - 3)-(x2 - 16, yoff + 35), RGB(0, 50, 10), BF
				Draw String (x1 + 80, yoff), car(n, i).name, RGB(200, 200, 200)
				nextx = x1 + 80 + 8 * Len(car(n, i).name)

				'Specify author if given
				If showauthors AndAlso Len(car(n, i).author) Then
					Draw String (nextx, yoff), " (" & car(n, i).author & ")", RGB(100, 100, 100)
					nextx += 8 * (3 + Len(car(n, i).author))
				End If
				
				'Show details
				s = UCase(car(n, i).id) & " - " & car(n, i).cfiles & " files - "
				l = Len(s)
				Draw String (x1 + 80, yoff + 16), s, RGB(100, 100, 100)

				nextx = x1 + 80 + 8 * l
				Select Case car(n, i).where
					Case 0
						Draw String (x1 + 80 + 8 * l, yoff + 16), "Stunts", RGB(100, 100, 0)
						nextx += 48
					Case Is > 0
						Draw String (x1 + 80 + 8 * l, yoff + 16), gdirname(car(n, i).where), RGB(0, 100, 0)
						nextx += 8 * Len(gdirname(car(n, i).where))
					Case -100
						Draw String (x1 + 80 + 8 * l, yoff + 16), "?", RGB(100, 100, 100)
						nextx += 8
					Case Else
						Draw String (x1 + 80 + 8 * l, yoff + 16), reponame(Abs(car(n, Abs(i)).where)), RGB(0, 100, 100)
						nextx += 8 * Len(reponame(Abs(car(n, Abs(i)).where)))
				End Select
			
				'Try to get and draw car icon
				Dim im As Any Ptr
				
				im = CarImage(car(n, i))
				If im Then
					Put (x1 + 8, yoff), im, Alpha
					ImageDestroy im
				End If
				
				yoff += 40
				If yoff >= y2 Then Exit For
			Next i
					
			'Only draw scroll bar if not all cars fit
			If panheight \ 40 < cars(n) Then
				Dim As Short barlength, barloc
				
				barlength = panheight * (panheight \ 40) / cars(n)
				barloc = panheight * (pan(n).first - 1) / (cars(n) - 1) + 48
				
				Line (x2 - 8, barloc + y1)-(x2 - 4, barloc + y1 + barlength), RGB(100, 100, 100), BF
			End If
			View
		Case 2
			If param > -1 AndAlso param <= gdirs + groups + repos Then Line (x1 + 8, 8 + 16 * param)-(x2 - 8, 23 + 16 * param), 0, BF
			yline = y1 + 8
			Draw String (x1 + 8, yline), Chr(7) & " Stunts", RGB(250, 250, 0)
			yline += 16
			For i As Short = 1 To gdirs
				Draw String (x1 + 8, yline), Chr(127) & " " & gdirname(i), RGB(0, 250, 0)
				yline += 16
			Next i
			For i As Short = 1 To repos
				Draw String (x1 + 8, yline), "@ " & reponame(i), RGB(0, 250, 250)
				yline += 16
			Next i
			For i As Short = 1 To groups
				Draw String (x1 + 8, yline), Chr(7) & " " & groupname(i), RGB(250, 100, 0)
				yline += 16
			Next i
		Case 3
			If Len(panelmessage) Then
				Dim As Short curx, cury, l
				Dim s As String
				
				curx = x1 + 8 : cury = y1 + 8
				Color RGB(&HAA, &HAA, &HAA)
				s = panelmessage
				While Len(s)
					l = InStr(s, "\n")
					If l Then
						Draw String (curx, cury), Left(s, l - 1)
						cury += 16 : curx = x1 + 8
						s = Mid(s, l + 2)
					Else
						Draw String (curx, cury), s
						s = ""
					End If
				WEnd
			End If
	End Select
	ScreenUnlock
End Sub


Function NameFromRes (resfile As String) As String
	Dim f As Integer, buf As String, p As Long, r As Short
	
	f = FreeFile
	Open resfile For Binary Access Read As f
	buf = Space(32)
	Get #f, 5, r
	Get #f, , buf
	p = InStr(buf, "gnam")
	p = ((p - 1) \ 4 + r) * 4 + 7 
	Get #f, p, p
	p += 7 + 8 * r
	buf = Space(40)
	Get #f, p, buf
	r = InStr(buf, Chr(0))
	buf = Left(buf, r - 1)
	Close f
	
	Return buf
End Function


Function ListName (id As Short) As String
	'Return the CRI list file name for a repo
	Dim s As String, c As String, n As Short
	
	For n = 1 To Len(reponame(id))
		c = LCase(Mid(reponame(id), n, 1))
		If (c >= "a" AndAlso c <= "z") OrElse (c >= "0" AndAlso c <= "9") Then
			 s &= c
			 If Len(s) = 8 Then Exit For
		End If
	Next n
	
	Return s
End Function


Sub CurlDown (webadd As String, finame As String)
	Dim f As Integer, s As String
	f = FreeFile
	Open Pipe "curl " & webadd & " -so " & finame For Input As f
	While Not EoF(f)
		Line Input #f, s
	WEnd
	Close f
End Sub


Sub LoadList (n As Byte, t As Byte, id As Byte)
	Dim source As String, s As String, s2 As String, tt As Byte
		
	'tt will be the "true t" for repositories, since their t becomes 2
	'when they actually contain an online group
	tt = t
	
	'Verify there's something in the list
	Select Case t
		Case -1
			'If configured to always update, then always download
			'the CRI if possible.
			If autoupdate = "always" Then
				UpdateRepo id
				If Not FileExists(source) Then Exit Sub
			Else
				'"yes" and "no" both update here only if the file does not
				'currently exist. The difference is that "yes" also updates once
				'a day forcefully.
				source = ListName(id) + ".cri"
				If Not FileExists(source) Then
					'"never" doesn't even update if there's no file
					If autoupdate <> "never" Then UpdateRepo id
					If Not FileExists(source) Then Exit Sub
				End If
			End If
			
			'Check to see if it's a repository or online group
			Dim f As Integer
			f = FreeFile
			Open source For Input As f
			Line Input #f, s
			s = LCase(Trim(s))
			If Left(s, 1) = "[" And Right(s, 1) = "]" Then
				'This is a true repository
				t = -1
			Else
				'This is an online group (hopefully)
				t = 2
			End If
			Close f
			
			pan(n).type = -1
			pan(n).name = reponame(id)
			pan(n).id = id
			pan(n).first = 1
		Case 0
			source = sdir
			pan(n).type = 0
			pan(n).name = "Stunts"
			pan(n).id = 0
			pan(n).first = 1
		Case 1
			If id > gdirs Then Exit Sub
			source = gdir(id)
			pan(n).type = 1
			pan(n).name = gdirname(id)
			pan(n).id = id
			pan(n).first = 1
		Case Else
			If id > groups Then Exit Sub
			pan(n).type = 2
			pan(n).name = groupname(id)
			pan(n).id = id
			pan(n).first = 1
	End Select
		
	'Load file list
	Dim thisid As String, i As Short
	
	Select Case t
		Case 0, 1
			Dim category As Byte
			
			cars(n) = 0
			s = Dir(source & SEP & ALLFILES)
			Do While Len(s)
				s2 = LCase(s)
				thisid = "" : category = 0
				If Left(s2, 3) = "car" And (Right(s2, 4) = ".res" Or Right(s2, 4) = ".ccf") Then
					thisid = Mid(s2, 4, 4)
					If Right(s2, 4) = ".res" Then category = 1 Else category = 2
				ElseIf Left(s2, 2) = "st" And (Right(s2, 4) = ".3sh" Or Right(s2, 4) = ".p3s") Then
					thisid = Mid(s2, 3, 4)
					category = 3
					'If Right(s2, 4) = ".3sh" Then category = 3
				ElseIf Left(s2, 3) = "std" And (Right(s2, 4) = ".vsh" Or Right(s2, 4) = ".pvs") Then
					thisid = Mid(s2, 5, 4)
				End If
				
				If Len(thisid) Then
					For i = 1 To cars(n)
						If car(n, i).id = thisid Then
							If car(n, i).cfiles < 8 Then
								car(n, i).cfiles += 1
								car(n, i).cfile(car(n, i).cfiles) = s
							End If
							Exit For
						End If
					Next i
					
					If i > cars(n) Then
						cars(n) += 1
						car(n, cars(n)).cfiles = 1
						car(n, cars(n)).cfile(1) = s
						car(n, cars(n)).id = thisid
						car(n, cars(n)).where = IIf(t, id, 0)
					End If
					
					If category = 1 Then
						car(n, i).paramfile = s
						car(n, i).name = NameFromRes(source & SEP & s)
					ElseIf category = 2 Then
						car(n, i).infofile = s
					ElseIf category = 3 Then
						car(n, i).shapefile = s
					End If
				End If
				s = Dir
			Loop
		Case 2
			Dim m As Short, found As Byte
			
			cars(n) = 0
			'If tt = -1, it'll keep the s read from the CRI above
			If tt = 2 Then s = group(pan(n).id)
						
			Do While Len(Trim(s))
				m = InStr(s, ",")
				If m Then
					s2 = LCase(Trim(Left(s, m - 1)))
					s = LTrim(Mid(s, m + 1))
				Else
					s2 = LCase(Trim(s))
					s = ""
				End If
				
				'Must be a valid car ID, 4 characters long
				If Len(s2) <> 4 Then Continue Do
				
				cars(n) += 1
				car(n, cars(n)).id = s2
				car(n, cars(n)).selected = 0
				car(n, cars(n)).cfiles = 0
				car(n, cars(n)).name = "Unknown car"
				car(n, cars(n)).author = ""
				car(n, cars(n)).where = -100
				car(n, cars(n)).infofile = ""
				
				'Try to find the car in Stunts
				found = 0
				s2 = Dir(sdir & SEP & ALLFILES)
				While Len(s2)
					If LCase(s2) = "car" & car(n, cars(n)).id & ".res" Then
						car(n, cars(n)).where = 0
						car(n, cars(n)).name = NameFromRes(sdir & SEP & s2)
						car(n, cars(n)).cfiles = 1
						car(n, cars(n)).cfile(1) = s2
						car(n, cars(n)).paramfile = s2
						found = -1
						Exit While
					End If
					s2 = Dir
				WEnd
				
				If found = 0 Then
					'Try to find it in garages
					For i = 1 To gdirs
						s2 = Dir(gdir(i) & SEP & ALLFILES)
						While Len(s2)
							If LCase(s2) = "car" & car(n, cars(n)).id & ".res" Then
								car(n, cars(n)).where = i
								car(n, cars(n)).name = NameFromRes(gdir(i) & SEP & s2)
								car(n, cars(n)).cfiles = 1
								car(n, cars(n)).cfile(1) = s2
								car(n, cars(n)).paramfile = s2
								found = -1
								Exit For
							End If
							s2 = Dir
						WEnd	
					Next i
				End If
			Loop
			
			'Find the other files
			For i = 1 To cars(n)
				If car(n, i).where Then
					s = Dir(gdir(car(n, i).where) & SEP & ALLFILES)
				Else
					s = Dir(sdir & SEP & ALLFILES)
				End If
				
				Do While Len(s)
					s2 = LCase(s)
					If s2 = "st" & car(n, i).id & ".3sh" Then
						car(n, i).cfiles += 1
						car(n, i).cfile(car(n, i).cfiles) = s
						car(n, i).shapefile = s
					ElseIf s2 = "car" & car(n, i).id & ".ccf" Then
						car(n, i).cfiles += 1
						car(n, i).cfile(car(n, i).cfiles) = s
						car(n, i).infofile = s
					ElseIf s2 = "st" & car(n, i).id & ".p3s" OrElse _
							s2 = "stda" & car(n, i).id & ".vsh" OrElse _
							s2 = "stdb" & car(n, i).id & ".vsh" OrElse _
							s2 = "stda" & car(n, i).id & ".pvs" OrElse _
							s2 = "stdb" & car(n, i).id & ".pvs" Then
					
						car(n, i).cfiles += 1
						car(n, i).cfile(car(n, i).cfiles) = s
					End If
					s = Dir
				Loop
			Next i
		Case -1		'Repository
			Dim f As Integer, m As Short, k As String, v As String
			
			f = FreeFile
			Open source For Input As f
			
			cars(n) = 0
			While Not EoF(f)
				Line Input #f, s
				s = Trim(s)
				If Left(s, 1) = "[" AndAlso Right(s, 1) = "]" Then
					cars(n) += 1
					car(n, cars(n)).id = LCase(Mid(s, 2, Len(s) - 2))
					car(n, cars(n)).cfiles = 0
					car(n, cars(n)).infofile = ""
					car(n, cars(n)).paramfile = ""
					car(n, cars(n)).shapefile = ""
					car(n, cars(n)).where = -id
					car(n, cars(n)).onlineloc = ""
					car(n, cars(n)).author = ""
					car(n, cars(n)).name = "Untitled"
				Else
					m = InStr(s, "=")
					If m Then
						k = LCase(Trim(Left(s, m - 1)))
						v = Trim(Mid(s, m + 1))
						
						Select Case k
							Case "name"
								If cars(n) Then car(n, cars(n)).name = v
							Case "author"
								If cars(n) Then car(n, cars(n)).author = v
							Case "zip"
								If cars(n) Then car(n, cars(n)).onlineloc = v
							Case "files[]"
								If cars(n) Then
									car(n, cars(n)).cfiles += 1
									car(n, cars(n)).cfile(car(n, cars(n)).cfiles) = v
									'Here, we might recognise the different file types,
									'but it really isn't very useful at this point, since
									'everything is zipped
								End If
						End Select
					End If
				End If
			WEnd
			
			Close f
	End Select
	
	'De-select all cars and see about author's name or other info
	For i As Short = 1 To cars(n)
		car(n, i).selected = 0
		
		'Try to gather author's name
		If Len(car(n, i).infofile) Then
			Select Case car(n, i).where
				Case 0 : source = sdir
				Case Is > 0 : source = gdir(car(n, i).where)
				Case Else : source = ""
			End Select
			
			If Len(source) AndAlso FileExists(source & SEP & car(n, i).infofile) Then
				Dim f As Integer, m As Short
				f = FreeFile
				Open source & SEP & car(n, i).infofile For Input As f
				While Not EoF(f)
					Line Input #f, s
					s = Trim(s)
					m = InStr(s, "=")
					If m = 0 Then m = InStr(s, ":")
					If m Then
						If LCase(RTrim(Left(s, m - 1))) = "author" Then
							car(n, i).author = LTrim(Mid(s, m + 1))
						End If
					End If
				WEnd
				Close f
			End If
		End If
	Next i

	SortList n, sortmethod
End Sub


Sub EditTitle (p As Byte, x1 As Short, y1 As Short, x2 As Short, y2 As Short)
	Dim As Byte editing, canedit(0 To 1), update = -1
	Dim fi(0 To 1) As String, col(0 To 1) As Long, posi(0 To 1) As Short
	
	Select Case pan(p).type
		Case 0		'Stunts - can only edit directory
			editing = 1
			canedit(0) = 0 : canedit(1) = -1
			fi(0) = "Stunts"
			fi(1) = sdir
			col(0) = RGB(250, 250, 0)
			col(1) = RGB(150, 150, 150)
		Case 1		'Garage - can edit everything
			editing = 0
			canedit(0) = -1 : canedit(1) = -1
			fi(0) = gdirname(pan(p).id)
			fi(1) = gdir(pan(p).id)
			col(0) = RGB(0, 250, 0)
			col(1) = RGB(150, 150, 150)
		Case 2		'Group - only name can be edited
			editing = 0
			canedit(0) = -1 : canedit(1) = 0
			fi(0) = groupname(pan(p).id)
			col(0) = RGB(250, 100, 0)
		Case -1		'Repo - can edit everything
			editing = 0
			canedit(0) = -1 : canedit(1) = -1
			fi(0) = reponame(pan(p).id)
			fi(1) = repo(pan(p).id)
			col(0) = RGB(0, 250, 250)
			col(1) = RGB(150, 150, 150)
	End Select
	
	posi(0) = Len(fi(0))
	posi(1) = Len(fi(1))
	
	Dim akey As String
	
	Do
		If update Then
			ScreenLock
			View Screen (x1 + 2, y1 + 2)-(x2 -2, y2 - 2)
			Line (x1 + 2, y1 + 2)-(x2 -2, y2 - 2), RGB(20, 20, 20), BF
			Draw String (x1 + 16, y1 + 8), fi(0), col(0)
			Draw String (x1 + 16, y1 + 24), fi(1), col(1)
			Line (x1 + 16 + 8 * posi(editing), y1 + 8 + 16 * editing)- Step (1, 13), col(editing), B
			View
			ScreenUnlock
			update = 0
		End If
	
		akey = InKey
		Select Case akey
			Case Chr(9)
				If canedit(1 - editing) Then
					editing = 1 - editing
					update = -1
				End If
			Case Chr(255, 72)
				If canedit(0) Then
					editing = 0
					update = -1
				End If
			Case Chr(255, 80)
				If canedit(1) Then
					editing = 1
					update = -1
				End If
			Case Chr(255, 75)
				If posi(editing) Then posi(editing) -= 1 : update = -1
			Case Chr(255, 77)
				If posi(editing) < Len(fi(editing)) Then posi(editing) += 1 : update = -1
			Case " " To "}"
				If Len(fi(editing)) < 100 Then
					fi(editing) = Left(fi(editing), posi(editing)) + akey + Mid(fi(editing), posi(editing) + 1)
					posi(editing) += 1
					update = -1
				End If
			Case Chr(8)
				If posi(editing) Then
					fi(editing) = Left(fi(editing), posi(editing) - 1) + Mid(fi(editing), posi(editing) + 1)
					posi(editing) -= 1
					update = -1
				End If
			Case Chr(255, 83)
				If posi(editing) < Len(fi(editing)) Then
					fi(editing) = Left(fi(editing), posi(editing)) + Mid(fi(editing), posi(editing) + 2)
					update = -1
				End If
			Case Chr(255, 71)
				posi(editing) = 0
				update = -1
			Case Chr(255, 79)
				posi(editing) = Len(fi(editing))
				update = -1
			Case Chr(27) : update = 0 : Exit Do
			Case Chr(13) : update = -1 : Exit Do
		End Select
	Loop
	
	If update Then
		'Assign fields back
		Select Case pan(p).type
			Case 0	'Stunts
				sdir = fi(1)
			Case 1	'Garage
				gdirname(pan(p).id) = fi(0)
				gdir(pan(p).id) = fi(1)
			Case 2	'Group
				groupname(pan(p).id) = fi(0)
			Case -1	'Repo
				reponame(pan(p).id) = fi(0)
				repo(pan(p).id) = fi(1)
		End Select
		pan(p).name = fi(0)
	End If
	
	DrawPanel p
	DrawPanel 2
End Sub


Sub Mousify
	Dim As Short i, j, carnum
	Dim As Byte shift, ctrl
	
	action = 0
	shift = MultiKey(&H2A) Or MultiKey(&h36)
	ctrl = MultiKey(&H1D)	
	GetMouse xm, ym, wm, bm
	
	For i = 0 To 3
		Dim As Short x1, y1, x2, y2
		
		Select Case i
			Case 0
				x1 = 0 : x2 = (swidth - 200) \ 2 - 8
				y1 = 0 : y2 = sheight - 108
			Case 1
				x1 = (swidth - 200) \ 2 : x2 = swidth - 208
				y1 = 0 : y2 = sheight - 108
			Case 2
				x1 = swidth - 200 : x2 = swidth - 1
				y1 = 0 : y2 = sheight - 108
			Case 3
				x1 = 0 : x2 = swidth - 1
				y1 = sheight - 100 : y2 = sheight -1
		End Select
		
		'Car lists
		If i = 0 OrElse i = 1 Then
			carnum = 0
			If xm >= x1 + 4 AndAlso xm <= x2 - 8 AndAlso ym >= y1 + 48 AndAlso ym <= y2 - 4 Then
				'If we click and hold the left button on a car list, this is
				'remembered until we change the button configuration
				If exbm = 0 AndAlso bm = 1 Then zone = i
				
				carnum = (ym - y1 - 48) \ 40 + pan(i).first
				
				'The right mouse button will be used for selecting
				If carnum <> 0 AndAlso carnum <= cars(i) AndAlso bm = 2 AndAlso exbm <> 2 Then
					'Selecting and deselecting cars, also wheeling up and down
					curpan = i
					If Not shift And Not ctrl Then
						For j = 1 To cars(0)
							car(0, j).selected = 0
						Next j
						For j = 1 To cars(1)
							car(1, j).selected = 0
						Next j
						car(i, carnum).selected = -1
					ElseIf ctrl Then
						For j = 1 To cars(1 - i)
							car(1 - i, j).selected = 0
						Next j
						car(i, carnum).selected = Not car(i, carnum).selected
					ElseIf shift Then
						Dim As Short first, last
						
						first = 0 : last = 0
						For j = 1 To cars(i)
							If car(i, j).selected Then
								last = j
								If first = 0 Then first = j
							End If
						Next j
						
						If first = 0 Then first = carnum
						If last = 0 Then last = carnum
						
						For j = 1 To cars(0)
							car(0, j).selected = 0
						Next j
						For j = 1 To cars(1)
							car(1, j).selected = 0
						Next j
						
						If carnum <= last Then
							first = carnum
						Else
							last = carnum
						End If
						
						For j = first To last
							car(i, j).selected = -1
						Next j
					End If
					DrawPanel 0 : DrawPanel 1
				ElseIf wm <> exwm AndAlso (y2 - y1 - 48) \ 40 < cars(i) Then
					'Wheel up and down
					pan(i).first += exwm - wm
					If pan(i).first <= 0 Then pan(i).first = 1
					DrawPanel i
				ElseIf bm = 0 AndAlso exbm = 1 AndAlso zone = 1 - i Then
					'Dropping cars from one side into the other
					Dim atleastone As Byte = 0
					For j = 1 To cars(1 - i)
						If car(1 - i, j).selected Then atleastone = -1 : Exit For
					Next j
					If atleastone Then
						'Tell the main loop we're trying to copy or move
						'cars and with which shift status
						action = 10 + Abs(shift <> 0) + 2 * Abs(ctrl <> 0)
					End If
				End If
			ElseIf xm >= x1 + 8 AndAlso xm <= x2 - 2 AndAlso ym >= y1 + 48 AndAlso ym <= y2 - 4 Then
				'Dragging scroll bars, which isn't working very well
				If bm = 1 AndAlso zone = -1 Then
					carnum = cars(i) * (ym - (y1 + 48)) / ((y2 - 4) - (y1 + 48))
					pan(i).first = carnum
					DrawPanel i
				End If
			ElseIf xm >= x1 + 4 AndAlso xm <= x2 - 4 AndAlso ym >= y1 + 4 AndAlso ym <= y1 + 40 Then
				'Click on list titles
				If bm = 2 Then EditTitle i, x1, y1, x2, y1 + 40
			End If
		ElseIf i = 2 Then
			If exbm = 0 AndAlso (bm = 1 OrElse bm = 2) Then
				If xm >= x1 + 8 AndAlso xm <= x2 - 8 AndAlso ym > y1 + 8 AndAlso ym < y1 + 8 + 16 * (gdirs + repos + groups + 1) Then
					Dim l As Short
					l = (ym - y1 - 8) \ 16
					Select Case l
						Case 0 : LoadList bm - 1, 0, 0 : DrawPanel bm - 1
						Case 1 To gdirs : LoadList bm - 1, 1, l : DrawPanel bm - 1
						Case gdirs + 1 To gdirs + repos
							LoadList bm - 1, -1, l - gdirs
							DrawPanel bm - 1
						Case gdirs + repos + 1 To gdirs + repos + groups
							LoadList bm - 1, 2, l - gdirs - repos
							DrawPanel bm - 1
					End Select
				End If
			ElseIf exbm = 0 AndAlso bm = 0 Then
				'Detect if we've moved among the group/garage list
				If xm >= x1 + 8 AndAlso xm <= x2 - 8 Then
					Dim num As Short
					num = (ym - y1 - 8) \ 16
					If num <> (exym - y1 - 8) \ 16 Then DrawPanel 2, num
				End If
			End If
		End If
	Next i
	
	exxm = xm : exym = ym : exwm = wm : exbm = bm
	If bm <> 1 Then zone = -1
End Sub


Sub SortList (which As Byte, by As Byte)
	Dim As Short i, j, k
	
	'Sort by selection
	For i = 1 To cars(which) - 1
		k = i
		For j = i + 1 To cars(which)
			If by Then	'Non-Zero: by ID
				If LCase(car(which, j).id) < LCase(car(which, k).id) Then k = j
			Else		'Zero: by name
				If LCase(car(which, j).name) < LCase(car(which, k).name) Then k = j
			End If
		Next j
		If k <> i Then Swap car(which, i), car(which, k)
	Next i
End Sub


Function SelectedCars(n As Byte) As Short
	Dim num As Short
	
	For i As Short = 1 To cars(n)
		If car(n, i).selected Then num += 1
	Next i
	
	Return num
End Function


Sub MoveSelected (force As Byte = 0)
	'Move selected cars between garages
	Dim As Short i, j
	Dim s As String, d As Byte
	
	'Determine destination
	If SelectedCars(0) Then d = 1 Else d = 0
	
	'Create a quick look-up string
	For i = 1 To cars(1 - d)
		If car(1 - d, i).selected Then s &= LCase(car(1 - d, i).id) & "/"
	Next i
	
	'Check that no car is overwritten
	If force = 0 Then
		For i = 1 To cars(d)
			If InStr(s, LCase(car(d, i).id)) Then
				'This car in destination also exists in source
				panelmessage = "Some cars would be overwritten!\nUse SHIFT to force"
				DrawPanel 3
				Exit Sub
			End If
		Next i
	End If
	
	'Check that, if destination is Stunts, no more than 32 cars
	If force = 0 Then
		If pan(d).type = 0 AndAlso cars(d) + SelectedCars(1 - d) > 32 Then
			panelmessage = "This would overflow Stunts directory!\nUse SHIFT to force"
			DrawPanel 3
			Exit Sub
		End If
	End If
	
	'Make sure if source is Stunts that the default car is not moved
	If force = 0 Then
		If pan(1 - d).type = 0 Then
			For i = 1 To cars(1 - d)
				If LCase(car(1 - d, i).id) = LCase(defaultcar) AndAlso car(1 - d, i).selected <> 0 Then
					panelmessage = "Moving " & UCase(defaultcar) & " would break Stunts!\nUse SHIFT to force"
					DrawPanel 3
					Exit Sub
				End If
			Next i
		End If
	End If
	
	'Proceed with move
	Dim As String sourcedir, destdir
	Dim As Short moved, overwritten
	
	If pan(1 - d).type = 0 Then sourcedir = sdir Else sourcedir = gdir(pan(1 - d).id)
	If pan(d).type = 0 Then destdir = sdir Else destdir = gdir(pan(d).id)
	
	If sourcedir = destdir Then
		panelmessage = "Source and destination are the same!"
		DrawPanel 3
		Exit Sub
	End If
	
	For i = 1 To cars(1 - d)
		If car(1 - d, i).selected Then
			For j = 1 To car(1 - d, i).cfiles
				FileCopy sourcedir & SEP & car(1 - d, i).cfile(j), destdir & SEP & car(1 - d, i).cfile(j)
				Kill sourcedir & SEP & car(1 - d, i).cfile(j)
			Next j
			moved += 1
		End If
	Next i
	
	LoadList 0, pan(0).type, pan(0).id
	LoadList 1, pan(1).type, pan(1).id
	DrawPanel 0 : DrawPanel 1
	panelmessage = moved & " cars moved"
	DrawPanel 3
End Sub


Sub AddToGroup
	'Add selected cars to group at destination
	Dim As Short i, num
	Dim s As String, d As Byte
	
	'Determine destination
	If SelectedCars(0) Then d = 1 Else d = 0
	
	'Add the cars
	For i = 1 To cars(1 - d)
		If car(1 - d, i).selected Then
			If InStr(LCase(group(pan(d).id)), LCase(car(1 - d, i).id)) = 0 Then
				If Len(group(pan(d).id)) Then group(pan(d).id) &= ","
				group(pan(d).id) &= LCase(car(1 - d, i).id)
				num += 1
			End If
		End If
	Next i
	LoadList d, 2, pan(d).id
	
	'Give result
	panelmessage = num & " cars added to group"
	DrawPanel d
	DrawPanel 3
End Sub


Sub DownloadCar (car As Car, t As Byte, id As UByte)
	'Download car from its "where" and place it in Stunts or a garage
	
	Dim s As String
	
	'Only valid from repositories
	If car.where >= 0 OrElse car.where = -100 Then Exit Sub
	
	Dim uz As String, n As Short, zipfile As String, target As String
	
	n = InStrRev(car.onlineloc, "/")
	If n = 0 Then Exit Sub	'Invalid web address
	zipfile = Mid(car.onlineloc, n + 1)
	
	CurlDown car.onlineloc, RAMdir & zipfile
	
	If t Then
		target = gdir(id)
	Else
		target = sdir
	End If
	
	uz = unzipcom
	n = InStr(uz, "$zipfile")
	If n Then uz = Left(uz, n - 1) + RAMdir & zipfile + Mid(uz, n + 8)
	n = InStr(uz, "$targetdir")
	If n Then uz = Left(uz, n - 1) + target + Mid(uz, n + 10)
	
	Dim f As Integer
	f = Freefile
	panelmessage = "" : n = 0
	Open Pipe uz For Input As f
	While Not EoF(f)
		Line Input #f, s
		s = Trim(s)
		If Len(s) Then
			If n > 100 Then
				panelmessage &= "\n"
				n = 0
			ElseIf Len(panelmessage) Then
				panelmessage &= " : "
				n += 3
			End If
			panelmessage &= s
			n += Len(s)
		End If
	WEnd
	Close f
	Kill RAMdir & zipfile
	
	If Len(panelmessage) Then
		DrawPanel 3
		Sleep 200
	End If
End Sub


Sub FeedGarageFromGroup (force As Byte = 0)
	'Move cars in group from wherever they are to a garage
	Dim As Short i, j
	Dim s As String, d As Byte
	
	'Determine destination
	If SelectedCars(0) Then d = 1 Else d = 0
	
	'Create a quick look-up string. In this case, with cars at destination
	For i = 1 To cars(d)
		s &= LCase(car(d, i).id) & "/"
	Next i
	
	'We don't check that no car is overwritten here. Instead, we just
	'move cars that don't exist at destination only
	
	'Check that, if destination is Stunts, no more than 32 cars
	If force = 0 Then
		If pan(d).type = 0 AndAlso cars(d) + SelectedCars(1 - d) > 32 Then
			panelmessage = "This would overflow Stunts directory!\nUse SHIFT to force"
			DrawPanel 3
			Exit Sub
		End If
	End If
	
	'Proceed with move
	Dim As String sourcedir, destdir
	Dim As Short moved, skipped
	
	If pan(d).type = 0 Then destdir = sdir Else destdir = gdir(pan(d).id)
	
	For i = 1 To cars(1 - d)
		If car(1 - d, i).selected Then
			If InStr(s, LCase(car(1 - d, i).id)) Then
				skipped += 1
			Else
				If car(1 - d, i).where = 0 Then
					sourcedir = sdir
				ElseIf car(1 - d, i).where >= 1 Then
					sourcedir = gdir(car(1 - d, i).where)
				Else
					'This would be to grab from repository. Not read yet
					'Be careful with where = -100, which stands for unknown car
					sourcedir = ""
					'DownloadCar car(1 - d, i), pan(d).type, pan(d).id
				EndIf
							
				If Len(sourcedir) Then
					For j = 1 To car(1 - d, i).cfiles
						FileCopy sourcedir & SEP & car(1 - d, i).cfile(j), destdir & SEP & car(1 - d, i).cfile(j)
						Kill sourcedir & SEP & car(1 - d, i).cfile(j)
					Next j
					moved += 1
				Else
					skipped += 1
				End If
			End If
		End If
	Next i
	
	LoadList 0, pan(0).type, pan(0).id
	LoadList 1, pan(1).type, pan(1).id
	DrawPanel 0 : DrawPanel 1
	panelmessage = moved & " cars moved"
	If skipped Then panelmessage &= ". " & skipped & " skipped."
	DrawPanel 3
End Sub


Sub DeleteCars (p As Byte, force As Byte = 0)
	Dim junk As Short = 0, i As Short, j As Short
	Dim n As Short = 0
	
	'Find if there is a junkyard garage
	For i = 1 To gdirs
		If Left(LCase(gdirname(i)), 4) = "junk" Then
			junk = i
			Exit For
		End If
	Next i
	
	'Count cars to delete
	For i = 1 To cars(p)
		If car(p, i).selected Then n += 1
	Next i
	If n = 0 Then Exit Sub	'No cars to delete, nothing to do
	
	'If deleting from Stunts, check that the default car is not deleted
	If force = 0 Then
		If pan(p).type = 0 Then
			For i = 1 To cars(p)
				If car(p, i).selected AndAlso LCase(car(p, i).id) = LCase(defaultcar) Then
					panelmessage = "Removing default car '" & UCase(defaultcar) & "' will break Stunts!\nUse SHIFT to force."
					DrawPanel 3
					Exit Sub
				End If
			Next i
		End If
	End If
	
	'If it corresponds, warn about deletion
	If junk = 0 OrElse junk = pan(p).id Then
		panelmessage = n & " cars will be deleted!\nPress Y to confirm or N to cancel"
		DrawPanel 3
		
		'Wait for a key for three seconds
		Dim t As Double, k As String
		t = Timer
		Do
			k = InKey
		Loop Until Timer >= t + 3 OrElse Len(k)
		If LCase(k) <> "y" Then
			panelmessage = "Cancelled deletion"
			DrawPanel 3
			Exit Sub
		End If
	End If
	
	'Proceed with deletion or moving cars to junkyard if any
	Dim sourcedir As String
	If pan(p).type = 0 Then sourcedir = sdir Else sourcedir = gdir(pan(p).id)
	
	For i = 1 To cars(p)
		If car(p, i).selected Then
			'If there's a junkyard, move cars there
			If junk <> 0 AndAlso pan(p).id <> junk Then
				For j = 1 To car(p, i).cfiles
					FileCopy sourcedir & SEP & car(p, i).cfile(j), gdir(junk) & SEP & car(p, i).cfile(j)
				Next j
			End If
			
			'Delete files from garage/Stunts
			For j = 1 To car(p, i).cfiles
				Kill sourcedir & SEP & car(p, i).cfile(j)
			Next j
		End If
	Next i
	
	'Output message with result
	panelmessage = n & " cars "
	If junk <> 0 AndAlso pan(p).id <> junk Then
		panelmessage &= "moved to " & gdirname(junk)
	Else
		panelmessage &= "permanently deleted"
	End If
	LoadList p, pan(p).type, pan(p).id
	LoadList 1 - p, pan(1 - p).type, pan(1 - p).id
	DrawPanel 0 : DrawPanel 1 : DrawPanel 3
End Sub


Sub RemoveCarsFromGroup (p As Byte)
	Dim r As Short, n As Short
		
	For i As Short = 1 To cars(p)
		If car(p, i).selected Then
			n = InStr(LCase(group(pan(p).id)), LCase(car(p, i).id))
			If n Then group(pan(p).id) = Left(group(pan(p).id), n - 1) + Mid(group(pan(p).id), n + 5)
		End If
	Next i
	
	If Right(group(pan(p).id), 1) = "," Then group(pan(p).id) = Left(group(pan(p).id), Len(group(pan(p).id)) - 1)
	
	LoadList p, 2, pan(p).id
	DrawPanel p
End Sub


Sub DeleteList (t As Byte, id As UByte)
	Dim i As Short
	
	Select Case t
		Case 1
			For i = id To gdirs - 1
				gdir(i) = gdir(i + 1)
				gdirname(i) = gdirname(i + 1)
			Next i
			gdirs -= 1
		Case 2
			For i = id To groups - 1
				group(i) = group(i + 1)
				groupname(i) = groupname(i + 1)
			Next i
			groups -= 1
		Case -1
			For i = id To repos - 1
				repo(i) = repo(i + 1)
				reponame(i) = reponame(i + 1)
			Next i
			repos -= 1
	End Select
End Sub


Sub FeedGarageFromRepo (force As Byte = 0)
	'Download cars from repository and put them in garage or Stunts
	Dim As Short i, j
	Dim s As String, d As Byte, dc As String
	Dim As Short copied, overwritten, failed
	
	'Determine destination
	If SelectedCars(0) Then d = 1 Else d = 0
	
	'Create a quick look-up string with selected source cars
	For i = 1 To cars(1 - d)
		If car(1 - d, i).selected Then s &= LCase(car(1 - d, i).id) & "/"
	Next i
	
	'Create one with cars at destination
	For i = 1 To cars(d)
		dc &= LCase(car(d, i).id) & "/"
	Next i
	
	'Check that no car is overwritten
	If force = 0 Then
		For i = 1 To cars(d)
			If InStr(s, LCase(car(d, i).id)) Then
				'This car in destination also exists in source
				panelmessage = "Some cars would be overwritten!\nUse SHIFT to force"
				DrawPanel 3
				Exit Sub
			End If
		Next i
	End If
	
	'Check that, if destination is Stunts, no more than 32 cars
	If force = 0 Then
		If pan(d).type = 0 AndAlso cars(d) + SelectedCars(1 - d) > 32 Then
			panelmessage = "This would overflow Stunts directory!\nUse SHIFT to force"
			DrawPanel 3
			Exit Sub
		End If
	End If
	
	Dim fs As String
	
	copied = 0
	For i = 1 To cars(1 - d)
		If car(1 - d, i).selected Then
			panelmessage = "Retrieving " & car(1 - d, i).name & "..."
			DrawPanel 3
			DownloadCar car(1 - d, i), pan(d).type, pan(d).id
			LoadList d, pan(d).type, pan(d).id
			DrawPanel d
			
			If InStr(dc, LCase(car(1 - d, i).id)) Then overwritten += 1
			
			If pan(d).type Then
				fs = Dir(gdir(pan(d).id) & SEP & ALLFILES)
			Else
				fs = Dir(sdir & SEP & ALLFILES)
			End If
			
			j = 0
			Do While Len(fs)
				If LCase(fs) = "car" & LCase(car(1 - d, i).id) & ".res" Then
					j = -1
					Exit Do
				End If
				
				fs = Dir
			Loop
			
			If j Then
				copied += 1
			Else
				failed += 1
			End If
		End If
	Next i
	panelmessage = copied & " cars downloaded"
	If failed Then panelmessage &= ". " & failed & " failed"
	If overwritten Then panelmessage &= ". " & overwritten & " overwritten"
	DrawPanel 3
	LoadList d, pan(d).type, pan(d).id
	DrawPanel d
End Sub


Sub SaveConfig
	Dim As String s, k, v
	Dim As Byte wstunts, wgars, wgroups, wrepos, wsize, wdef, lu, au
	Dim n As Short, i As Short
	
	Open RAMdir & "pgarage.cfg" For Input As 1
	Open RAMdir & "temp.cfg" For Output As 2
	
	While Not EoF(1)
		Line Input #1, s
		If Left(Trim(s), 1) = "#" Then
			Print #2, s
		Else
			n = InStr(s, "=")
			If n = 0 Then
				Print #2, s
			Else
				k = LCase(Trim(Left(s, n - 1)))
				v = Trim(Mid(s, n + 1))
				
				Select Case k
					Case "stunts", "stuntsdir", "sdir"
						If Not wstunts Then
							Print #2, "stunts=" & sdir
							wstunts = -1
						End If
					Case "garage", "gdir", "gar"
						If Not wgars Then
							For i = 1 To gdirs
								Print #2, "garage=" & gdirname(i) & "|" & gdir(i)
							Next i
							wgars = -1
						End If
					Case "repo", "repository"
						If Not wrepos Then
							For i = 1 To repos
								Print #2, "repo=" & reponame(i) & "|" & repo(i)
							Next i
							wrepos = -1
						End If
					Case "group"
						If Not wgroups Then
							For i = 1 To groups
								Print #2, "group=" & groupname(i) & "|" & group(i)
							Next i
							wgroups = -1
						End If
					Case "defaultcar"
						If Not wdef Then
							Print #2, "defaultcar=" & defaultcar
							wdef = -1
						End If
					Case "width", "height"
						If Not wsize Then
							Print #2, "width=" & swidth
							Print #2, "height=" & sheight
							wsize = -1
						End If
					Case "lastupdate"
						If Not lu Then
							Print #2, "lastupdate=" & Right(Date, 4) & "-" & Left(Date, 2) & "-" & Mid(Date, 4, 2)
							lu = -1
						End If
					Case "autoupdate"
						If Not au Then
							Print #2, "autoupdate=" & autoupdate
							au = -1
						End If
					Case Else
						Print #2, s
				End Select
			End If
		End If
	WEnd
	
	If Not wstunts Then Print #2, "stunts=" & sdir
	If Not wdef Then Print #2, "defaultcar=" & defaultcar
	If Not wgars Then
		For i = 1 To gdirs
			Print #2, "garage=" & gdirname(i) & "|" & gdir(i)
		Next i
	End If
	If Not wrepos Then
		For i = 1 To repos
			Print #2, "repo=" & reponame(i) & "|" & repo(i)
		Next i
	End If
	If Not wgroups Then
		For i = 1 To groups
			Print #2, "group=" & groupname(i) & "|" & group(i)
		Next i
	End If
	If Not lu Then Print #2, "lastupdate=" & ISODate
	If Not au Then Print #2, "autoupdate=" & autoupdate
	
	Close 2, 1
	
	Kill RAMdir & "pgarage.cfg"
	Name RAMdir & "temp.cfg", RAMdir & "pgarage.cfg"
End Sub


Sub UpdateRepo (id As Byte = 0)
	'Attempt to download CRI file from repository
	
	Dim ln As String
	
	'If repo number is specified, only update that one. Otherwise, update all
	If id Then
		panelmessage = "Retrieving list for " & reponame(id) & "..."
		DrawPanel 3
		
		If FileExists("tempcri") Then Kill "tempcri"
		ln = RAMdir & ListName(id) + ".cri"
		CurlDown repo(id), RAMdir & "tempcri"
		If FileExists(RAMdir & "tempcri") AndAlso FileLen(RAMdir & "tempcri") <> 0 Then
			If FileExists(ln) Then Kill ln
			Name RAMdir & "tempcri", ln
			panelmessage = "Download successful"
		Else
			panelmessage = "Download failed"
		End If
		DrawPanel 3
	Else
		For i As Short = 1 To repos
			panelmessage = "Retrieving list for " & reponame(i) & "..."
			DrawPanel 3
			
			If FileExists("tempcri") Then Kill "tempcri"
			ln = RAMdir & ListName(i) + ".cri"
			CurlDown repo(i), RAMdir & "tempcri"
			If FileExists(RAMdir & "tempcri") AndAlso FileLen(RAMdir & "tempcri") <> 0 Then
				If FileExists(ln) Then Kill ln
				Name RAMdir & "tempcri", ln
			Else
				panelmessage = "Download failed"
				DrawPanel 3
				Sleep 500
			End If
		Next i
		panelmessage = ""
		DrawPanel 3
	End If
End Sub


Sub ApplyFilter(filter As String, op As String, kind As String)
	'If filter contains commas, apply recursively
	Dim n As Short
	n = InStr(filter, ",")
	If n Then
		Dim As String subfilter, filterstring, subop
		subop = op
		filterstring = filter
		Do
			subfilter = Left(filterstring, n - 1)
			filterstring = Mid(filterstring, n + 1)
			ApplyFilter subfilter, subop, kind
			
			'For "select", when recursive, only the first one is "select" and
			'then, it's "or". Otherwise, it's pretty useless
			If subop = "select" Then subop = "or"
			n = InStr(filterstring, ",")
		Loop Until n = 0
		If Len(filterstring) Then ApplyFilter filterstring, subop, kind
		Exit Sub
	End If
	
	'Calculate target panel and how many cars are selected
	Dim d As Byte, numsel As Short
	numsel = SelectedCars(0)
	If numsel Then
		d = 0
	Else
		numsel = SelectedCars(1)
		If numsel Then
			d = 1
		Else
			d = curpan
		End If
	End If
	
	Dim xfield As String, q As Byte
	
	For i As Short = 1 To cars(d)
		Select Case kind
			Case "name" : xfield = car(d, i).name
			Case "author" : xfield = car(d, i).author
			Case "id" : xfield = car(d, i).id
		End Select
		
		q = 0
		If (Left(filter, 1) = "*" AndAlso Right(filter, 1) = "*") Then
			If InStr(LCase(xfield), LCase(Mid(filter, 2, Len(filter) - 2))) Then q = -1
		ElseIf InStr(filter, "*") = 0 Then
			If InStr(LCase(xfield), LCase(filter)) Then q = -1
		ElseIf Left(filter, 1) = "*" Then
			If Right(LCase(xfield), Len(filter) - 1) = LCase(Mid(filter, 2)) Then q = -1
		Else
			If Left(LCase(xfield), Len(filter) - 1) = LCase(Left(filter, Len(filter) - 1)) Then q = -1
		End If
		
		Select Case op
			Case "select" :	car(d, i).selected = q
			Case "and" : If q = 0 Then car(d, i).selected = 0
			Case "or" : If q Then car(d, i).selected = -1
			Case "not" : car(d, i).selected = Not car(d, i).selected
		End Select
	Next i
	
	DrawPanel d
End Sub


Sub SetCarsAuthor (author As String)
	Dim r As Byte
	If SelectedCars(0) Then
		r = 0
	ElseIf SelectedCars(1) Then
		r = 1
	Else
		Exit Sub
	End If
	
	For i As Short = 1 To cars(r)
		If car(r, i).selected Then
			car(r, i).author = Trim(author)
			SaveCarConfig car(r, i)
		End If
	Next i
	
	DrawPanel r
End Sub


Sub CommandLine
	Dim As String lin, akey, c
	Dim As Short x1, x2, y1, y2, x, n
	Dim As Byte update = -1
	
	x1 = 0 : x2 = swidth - 1
	y1 = sheight - 100 : y2 = sheight -1
	
	panelmessage = ""
	DrawPanel 3
	
	Do
		If update Then
			ScreenLock
			Line (x1 + 16, y1 + 8)-(x2 - 16, y1 + 23), RGB(20, 20, 20), BF
			Draw String (x1 + 16, y1 + 8), "@>", RGB(180, 180, 180)
			Draw String (x1 + 16 + 16, y1 + 8), lin, RGB(120, 120, 120)
			Line (x1 + 16 + 16 + 8 * x, y1 + 10)- Step (1, 11), RGB(180, 180, 180), B
			ScreenUnlock
			update = 0
		End If
		
		akey = InKey

		Select Case akey
			Case " " To Chr(126)
				If Len(lin) < (x2 - x1) \ 8 - 8 Then
					lin = Left(lin, x) & akey & Mid(lin, x + 1)
					x += 1
					update = -1
				End If
			Case Chr(8)
				If x Then
					lin = Left(lin, x - 1) & Mid(lin, x + 1)
					x -= 1
					update = -1
				End If
			Case Chr(255, 75)
				If x Then
					x -= 1
					update = -1
				End If
			Case Chr(255, 77)
				If x < Len(lin) Then
					x += 1
					update = -1
				End If
			Case Chr(255, 71)
				x = 0 : update = -1
			Case Chr(255, 79)
				x= Len(lin) : update = -1
			Case Chr(13) : Exit Do
			Case Chr(27) : lin = "" : Exit Do
		End Select
	Loop
	
	panelmessage = ""
	If Len(lin) Then
		lin = Trim(lin)
		n = InStr(lin, " ")
		If n Then
			c = LCase(Trim(Left(lin, n - 1)))
			lin = Trim(Mid(lin, n + 1))
		Else
			c = LCase(Trim(lin))
			lin = ""
		End If
	End If
	
	If Len(c) Then
		Dim carg(1 To 5) As String, cargs As Byte, lintemp As String
		lintemp = lin
		Do
			n = InStr(lintemp, " ")
			If n Then
				cargs += 1
				carg(cargs) = Left(lintemp, n - 1)
				lintemp = LTrim(Mid(lintemp, n + 1))
			ElseIf Len(lintemp) Then
				cargs += 1
				carg(cargs) = lintemp
				lintemp = ""
			End If
		Loop Until Len(lintemp) = 0 OrElse cargs = 5
		
		Select Case c
			Case "select", "and", "or", "not"
				If InStr("/name/author/id/", "/" & LCase(carg(1)) & "/") Then
					ApplyFilter carg(2), c, LCase(carg(1))
				Else
					ApplyFilter carg(1), c, "name"
				End If
				panelmessage = "Applied filter"
			Case "setauthor"
				If Len(Trim(lin)) Then
					If SelectedCars(0) Or SelectedCars(1) Then
						SetCarsAuthor Trim(lin)
						panelmessage = "Changed cars' author to " & Trim(lin)
					Else
						panelmessage = "No cars selected to change author name"
					End If
				Else
					panelmessage = "No author name given"
				End If
			Case "newgarage"
				If gdirs < UBound(gdir) Then
					gdirs += 1
					If Len(carg(1)) Then
						gdirname(gdirs) = carg(1)
					Else
						gdirname(gdirs) = "Garage #" & gdirs
					End If
					If Len(carg(2)) Then
						gdir(gdirs) = carg(2)
					Else
						#ifdef __FB_LINUX__
							gdir(gdirs) = Environ("HOME")
						#elseif defined(__FB_DOS__)
							gdir(gdirs) = ExePath
						#else
							gdir(gdirs) = Environ("HOMEDRIVE") & Environ("HOMEPATH")
						#endif
					End If
				Else
					panelmessage = "You already have too many entries"
				End If
			Case Else : panelmessage = "Unknown command: " & c
		End Select
	End If
	
	DrawPanel 3
End Sub


Initialise
LoadConfig
DetectGoodDirs

ScreenRes swidth, sheight, 32
Width swidth \ 8, sheight \ 16
WindowTitle "Pretty Garage 1.2"

LoadList 0, 0, 0
If gdirs Then
	LoadList 1, 1, 1
Else
	LoadList 1, 0, 0
End If

panelmessage = "Welcome to Pretty Garage - release date " & __DATE_ISO__
panelmessage &= "\nThis is free software under GPLv3. See lincese.txt"
panelmessage &= "\nCopyright 2024"
If Left(__DATE_ISO__, 4) <> "2024" Then panelmessage &= "-" & Left(__DATE_ISO__, 4)
panelmessage &= " Lucas Pedrosa"
For i As Byte = 0 To 3
	DrawPanel i
Next i

If autoupdate = "always" OrElse (autoupdate = "yes" AndAlso ISODate <> lastupdate) Then
	UpdateRepo
End If

Dim akey As String

Do
	Sleep 1
	action = 0
	Mousify
	
	akey = InKey
	Select Case akey
		Case Chr(255, 72)
			If pan(curpan).first > 1 Then
				pan(curpan).first -= 1
				DrawPanel curpan
			End If
		Case Chr(255, 80)
			If pan(curpan).first < cars(curpan) Then
				pan(curpan).first += 1
				DrawPanel curpan
			End If
		Case Chr(9)
			curpan = 1 - curpan
			DrawPanel 0 : DrawPanel 1
		Case Chr(255, 83)	'Delete
			action = 20
		Case Chr(1)		'Ctrl+A - Select/deselect all
			action = 30
		Case Chr(4)		'Ctrl+D - Delete group/garage/repository
			action = 40
		Case Chr(19)	'Ctrl+S - Change sorting method
			'This could be changed to only re-sort the current list
			sortmethod = 1 - sortmethod
			LoadList 0, pan(0).type, pan(0).id
			LoadList 1, pan(1).type, pan(1).id
			panelmessage = "Sorting car lists by car " & IIf(sortmethod, "ID", "name")
			DrawPanel 0 : DrawPanel 1 : DrawPanel 3
		Case Chr(27), Chr(255) & "k" : SaveConfig : Exit Do
		Case Chr(17)
			panelmessage = "Exit without saving configuration? Y/N"
			DrawPanel 3
			
			Dim t As Double
			t = Timer
			Do
				akey = InKey
			Loop Until Timer >= t + 3 OrElse Len(akey)
			If LCase(akey) = "y" Then Exit Do
			panelmessage = ""
			DrawPanel 3
		Case Chr(6)	'Ctrl+F - Force regenerate image
			Dim d As Byte
			If SelectedCars(0) Then
				d = 0
			ElseIf SelectedCars(1) Then
				d = 1
			Else
				d = -1
			End If
			If d >= 0 Then
				For i As Short = 1 To cars(d)
					If car(d, i).selected Then
						Dim t As Double
						t = Timer
						Do
							akey = InKey
						Loop Until Timer > t + 1 OrElse ValInt(akey)
						If ValInt(akey) = 0 Then akey = "1"
						Dim p As Any Pointer
						p = CarImage(car(d, i), -1, ValInt(akey))
						ImageDestroy p
						DrawPanel d
						Exit For
					End If
				Next i
			End If
		Case Chr(7)	'Ctrl+G
			Dim r As Byte
			If SelectedCars(0) = 1 Then
				r = 0
			ElseIf SelectedCars(1) = 1 Then
				r = 1
			Else
				r = -1
			End If
			
			If r >= 0 Then
				For i As Short = 1 To cars(r)
					If car(r, i).selected Then
						Locate 1, 1 : Print ">" & car(r, i).author
						Line Input car(r, i).author
						SaveCarConfig car(r, i)
						Exit For
					End If
				Next i
			End If
			
			DrawPanel 0
		Case Chr(13)
			CommandLine
	End Select
	
	Select Case action
		Case 10		'Move cars between garages
			'Destination
			Dim d As Byte
			
			If SelectedCars(0) Then d = 1 Else d = 0
			If (pan(1 - d).type = 0 OrElse pan(1 - d).type = 1) _
				AndAlso (pan(d).type = 0 OrElse pan(d).type = 1) Then
				'This is, move between garages
				MoveSelected
			ElseIf pan(d).type = 2 Then
				'No matter source, this is: add to group
				AddToGroup
			ElseIf pan(1 - d).type = 2 AndAlso (pan(d).type = 0 OrElse pan(d).type = 1) Then
				'Copy from group to garage, that is, from wherever the
				'cars in the group are
				FeedGarageFromGroup
			ElseIf pan(1 - d).type = -1 AndAlso (pan(d).type = 0 OrElse pan(d).type = 1) Then
				'Download from repository into garage or Stunts
				If car(1 - d, 1).where < 0 Then
					FeedGarageFromRepo
				Else
					FeedGarageFromGroup
				End If
			End If
		Case 11		'Force move cars (even if overwriting or too many for Stunts)
			'Destination
			Dim d As Byte
			
			If SelectedCars(0) Then d = 1 Else d = 0
			If (pan(1 - d).type = 0 OrElse pan(1 - d).type = 1) _
				AndAlso (pan(d).type = 0 OrElse pan(d).type = 1) Then
				'This is, move between garages
				MoveSelected -1
			ElseIf pan(d).type = 2 Then
				'No matter source, this is: add to group
				AddToGroup
			ElseIf pan(1 - d).type = 2 AndAlso (pan(d).type = 0 OrElse pan(d).type = 1) Then
				'Copy from group to garage, that is, from wherever the
				'cars in the group are
				FeedGarageFromGroup -1
			ElseIf pan(1 - d).type = -1 AndAlso (pan(d).type = 0 OrElse pan(d).type = 1) Then
				'Download from repository into garage or Stunts
				If car(1 - d, 1).where < 0 Then
					FeedGarageFromRepo -1
				Else
					FeedGarageFromGroup
				End If
			End If
		Case 12		'Make new group out of set of cars
			If groups < UBound(group) Then
				'Create the group
				groups += 1
				group(groups) = ""
				groupname(groups) = "Group #" & groups
				
				'Destination
				Dim d As Byte
			
				If SelectedCars(0) Then d = 1 Else d = 0
				
				pan(d).type = 2
				pan(d).id = groups
				pan(d).n = 0
				pan(d).first = 1
				pan(d).name = groupname(groups)
				
				AddToGroup
				DrawPanel 2
			End If
		Case 20		'Delete cars
			Dim d As Byte
			If SelectedCars(0) Then
				d = 0
			ElseIf SelectedCars(1) Then
				d = 1
			Else
				d = -1
			End If
			If d >= 0 Then
				If pan(d).type = 0 OrElse pan(d).type = 1 Then
					DeleteCars d, MultiKey(&H2A) Or MultiKey(&h36)
				ElseIf pan(d).type = 2 Then
					RemoveCarsFromGroup d
				End If
			End If
		Case 30		'Select all
			Dim d As Byte, q As Byte = 0
			
			If SelectedCars(0) Then
				d = 0
			ElseIf SelectedCars(1) Then
				d = 1
			Else
				d = curpan
			End If
			
			For i As Short = 1 To cars(d)
				If car(d, i).selected = 0 Then q = -1
				car(d, i).selected = -1
			Next i
			
			If q = 0 Then
				For i As Short = 1 To cars(d)
					car(d, i).selected = 0
				Next i
				panelmessage = "Cleared selection"
			Else
				panelmessage = "All cars in list selected"
			End If
			
			DrawPanel d : DrawPanel 3
		Case 40		'Delete group/garage/repository
			Dim d As Byte
			d = curpan
			If pan(d).type Then
				panelmessage = "Confirm deletion of "
				Select Case pan(d).type
					Case 1
						panelmessage &= "garage " & gdirname(pan(d).id) & "?"
						panelmessage &= "\nThis will not remove the files"
					Case 2 : panelmessage &= "group " & groupname(pan(d).id) & "?"
					Case -1 : panelmessage &= "repository " & reponame(pan(d).id) & "?"
				End Select
				DrawPanel 3
				Dim t As Double
				t = Timer
				Do
					akey = InKey
				Loop Until Len(akey) <> 0 OrElse Timer >= t + 3
				If UCase(akey) = "Y" Then
					DeleteList pan(d).type, pan(d).id
					panelmessage = "List deleted."
					DrawPanel 2 : DrawPanel 3
					LoadList d, 0, 0
					DrawPanel d
				Else
					panelmessage = "Cancelled deletion."
					DrawPanel 3
				End If
			End If
	End Select
Loop
