' targalib.bi is part of CarWorks (car editor for Stunts)
' Copyright (C) 2021-2022  Lucas Pedrosa

' CarWorks is free software: you can redistribute it and/or modify
' it under the terms of the GNU General Public License as published by
' the Free Software Foundation, version 3 of the License.

' This program is distributed in the hope that it will be useful,
' but WITHOUT ANY WARRANTY; without even the implied warranty of
' MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
' GNU General Public License for more details.

' You should have received a copy of the GNU General Public License
' along with this program.  If not, see <http://www.gnu.org/licenses/>.

#include once "file.bi"
#define MAX_TARGA_WIDTH 8192
#define MAX_TARGA_HEIGHT 8192
#macro c16to15(x)
	(x And &B11111) Or ((x ShR 1) And &B1111111111100000) Or &H8000
#endmacro
#macro c15to16(x)
	(x And &B11111) Or ((x And &B0111111111100000) ShL 1)
#endmacro

Type TargaHeader Field = 1
	IDlength As UByte
	ColorMapType As UByte
	ImageType As UByte
	ColorMapStart As UShort
	ColorMapLength As UShort
	ColorMapDepth As UByte
	XOffset As UShort
	YOffset As UShort
	ImageWidth As UShort
	ImageHeight As UShort
	PixelDepth As UByte
	ImageDescriptor As UByte
End Type

Dim Shared TargaError As Short, TargaErrorMessage As String


Function Image16to32(img As Any Pointer) As Any Pointer
	Dim img2 As Any Ptr
	Dim source As UShort Ptr, dest As ULong Ptr
	Dim As Long w, h, spitch, dpitch
	Dim As Any Ptr sstart, dstart
	Dim As UByte r, g, b
	
	ImageInfo img, w, h, , spitch, sstart
	img2 = ImageCreate(w, h, &HFF000000, 32)
	ImageInfo img2, , , , dpitch, dstart
	
	For j As Short = 0 To h - 1
		source = sstart + j * spitch
		dest = dstart + j * dpitch
		For i As Short = 0 To w - 1
			b = (source[i] And 31) ShL 3
			g = ((source[i] ShR 5) And 63) ShL 2
			r = ((source[i] ShR 11) And 31) ShL 3
			dest[i] = RGB(r, g, b)
		Next i
	Next j
	
	Return img2
End Function


Function Image32to16(img As Any Pointer) As Any Pointer
	Dim img2 As Any Ptr
	Dim source As ULong Ptr, dest As UShort Ptr
	Dim As Long w, h, spitch, dpitch
	Dim As Any Ptr sstart, dstart
	Dim As UByte r, g, b
	
	ImageInfo img, w, h, , spitch, sstart
	img2 = ImageCreate(w, h, &HFF000000, 16)
	ImageInfo img2, , , , dpitch, dstart
	
	For j As Short = 0 To h - 1
		source = sstart + j * spitch
		dest = dstart + j * dpitch
		For i As Short = 0 To w - 1
			b = (source[i] And 255) ShR 3
			g = ((source[i] ShR 8) And 255) ShR 2
			r = ((source[i] ShR 16) And 255) ShR 3
			dest[i] = b Or (g ShL 5) Or (r ShL 11)
		Next i
	Next j
	
	Return img2
End Function


Function Image8to32(img As Any Ptr, pal As Any Ptr) As Any Pointer
	Dim img2 As Any Ptr, palp As Short
	Dim source As UByte Ptr, dest As ULong Ptr, bpal As UByte Ptr
	Dim As Long w, h, spitch, dpitch
	Dim As Any Ptr sstart, dstart
	Dim As UByte r, g, b
	
	bpal = pal
	
	ImageInfo img, w, h, , spitch, sstart
	img2 = ImageCreate(w, h, &HFF000000, 32)
	ImageInfo img2, , , , dpitch, dstart
	
	For j As Short = 0 To h - 1
		source = sstart + j * spitch
		dest = dstart + j * dpitch
		For i As Short = 0 To w - 1
			palp = 3 * source[i]
			r = bpal[palp + 2]
			g = bpal[palp + 1]
			b = bpal[palp]
			dest[i] = RGB(r, g, b)
		Next i
	Next j
	
	Return img2
End Function


Function Image32to8(img As Any Ptr, pal As Any Ptr) As Any Ptr
	Dim img2 As Any Ptr
	Dim source As ULong Ptr, dest As UByte Ptr
	Dim As Long w, h, spitch, dpitch
	Dim As Any Ptr sstart, dstart
	Dim As UByte r, g, b
	Dim bpal As UByte Ptr
	Dim dist As Short, mindist As Short, mincol As UByte
	Dim As Short i, j, n
	
	ImageInfo img, w, h, , spitch, sstart
	img2 = ImageCreate(w, h, 0, 8)
	ImageInfo img2, , , , dpitch, dstart
	
	For j = 0 To h - 1
		source = sstart + j * spitch
		dest = dstart + j * dpitch
		For i = 0 To w - 1
			b = source[i] And 255
			g = (source[i] ShR 8) And 255
			r = (source[i] ShR 16) And 255
			
			'Find closest colour
			bpal = pal : mindist = 800
			For n = 0 To 255
				dist = Abs(bpal[0] - b) + Abs(bpal[1] - g) + Abs(bpal[2] - r)
				If dist < mindist Then
					mindist = dist
					mincol = n
				End If
				bpal += 3
			Next n
			dest[i] = mincol
		Next i
	Next j
	
	Return img2
End Function


Function GetGeneral8bitPalette As Any Ptr
	Dim pal As UByte Ptr
	Dim As Byte r, g, b
	Dim n As Short
	
	pal = Allocate(768)
	
	n = 0
	For b = 0 To 3
		For g = 0 To 7
			For r = 0 To 7
				pal[n] = 85 * b
				pal[n + 1] = 36.4 * g
				pal[n + 2] = 36.4 * r
				n += 3
			Next r
		Next g
	Next b
	
	Return pal
End Function


Function TargaLoad(filename As String, pal As UByte Ptr = 0) As Any Pointer
	'Reset error
	TargaError = 0 : TargaErrorMessage = ""
	
	If Not FileExists(filename) Then
		TargaError = 201
		TargaErrorMessage = "File not found"
		Return 0
	End If

	Dim h As TargaHeader, f As Short
	Dim iptr As UByte Ptr	'Pointer to new image being generated
	Dim linelength As Long	'Image line length within iptr in bytes
	Dim imagestart As UByte Ptr	'Pointer to start of image data in iptr
	Dim idatastart As Long	'File pointer to start of image data
	Dim alphachannel As Byte	'Whether there's an alpha channel
	Dim inverted As Byte	'Whether the image goes from bottom to top
	Dim buffer As UByte Ptr		'Where compressed data will be loaded
	Dim wp As Long		'Current write pointe (to image)
	Dim rp As Long		'Current read pointer (from buffer)
	
		
	'Open the file
	f = FreeFile
	Open filename For Binary Access Read As f
	Get #f, 1, h	'Load the header
	
	'Safety maximums have been set
	'You can change these maximums to your preference
	If h.ImageWidth > MAX_TARGA_WIDTH Or h.ImageHeight > MAX_TARGA_HEIGHT Then
		TargaError = 202
		TargaErrorMessage = "Image is too large"
		Close f : Return 0
	End If
	
	'Calculate where image data starts in the file
	idatastart = 19 + h.IDlength
	If h.ColorMapType Then idatastart += h.ColorMapLength * (h.ColorMapDepth \ 8)
	
	'See if image is inverted
	If (h.ImageDescriptor And 32) = 0 Then inverted = -1
	If (h.ImageDescriptor And 16) <> 0 Then
		'X is inverted. Very unusual thing. Not currently supported
		TargaError = 203
		TargaErrorMessage = "Format not supported: X axis is inverted"
		Close f : Return 0
	End If
	
	'See if there's an alpha channel
	If h.ImageDescriptor And 15 Then alphachannel = -1
	
	'Create a buffer of image size
	If h.PixelDepth = 8 Or h.PixelDepth = 16 Then
		iptr = ImageCreate(h.ImageWidth, h.ImageHeight, , h.PixelDepth)
	Else
		iptr = ImageCreate(h.ImageWidth, h.ImageHeight, , 32)
	End If
	ImageInfo iptr, , , , linelength, imagestart

	'Load all image data from file to a buffer
	buffer = Allocate(LOF(f) - idatastart + 1)
	If buffer = 0 Then
		'Could not allocate memory to load the file
		TargaError = 204
		TargaErrorMessage = "Could not allocate a buffer to decompress image"
		ImageDestroy iptr
		Close f
		Return 0
	End If
	Get #f, idatastart, *buffer, LOF(f) - idatastart + 1
	
	'==== Test for each of the possible supported formats ====
	
	'True colour, RLE compressed, with alpha channel
	If h.PixelDepth = 32 And h.ImageType = 10 And alphachannel <> 0 Then
		Dim As UByte copying, repeating, sample(0 To 3)
		Dim column As Long
		
		rp = 0	'Reset read pointer
		
		'For every line
		For i As Long = 0 To h.ImageHeight - 1
			'Calculate relative start of the line
			If inverted Then
				wp = (h.ImageHeight - i - 1) * linelength
			Else
				wp = i * linelength
			End If
			
			'Starting the line from column zero
			column = 0
			Do
				'Check to see if there's RLE or non-RLE pending
				If repeating Then
					'Repeat one more sample of RLE block
					For j As Byte = 0 To 3
						imagestart[wp + j] = sample(j)
					Next j
					wp += 4
					column += 1
					repeating -= 1
				ElseIf copying Then
					'Copy one more sample from buffer
					For j As Byte = 0 To 3
						imagestart[wp + j] = buffer[rp + j]
					Next j
					wp += 4 : rp += 4
					column += 1
					copying -= 1
				Else
					'Process an RLE/non-RLE header
					copying = buffer[rp]
					rp += 1
					If copying And 128 Then
						repeating = copying - 127
						copying = 0
						For j As Byte = 0 To 3
							sample(j) = buffer[rp + j]
						Next j
						rp += 4
					Else
						copying += 1
					End If
				End If
			Loop Until column = h.ImageWidth
		Next i
	'True colour, RLE compressed, without alpha channel
	ElseIf h.PixelDepth = 24 And h.ImageType = 10 And alphachannel = 0 Then
		Dim As UByte copying, repeating, sample(0 To 3)
		Dim column As Long
		
		rp = 0	'Reset read pointer
		
		'For every line
		For i As Long = 0 To h.ImageHeight - 1
			'Calculate relative start of the line
			If inverted Then
				wp = (h.ImageHeight - i - 1) * linelength
			Else
				wp = i * linelength
			End If
			
			'Starting the line from column zero
			column = 0
			Do
				'Check to see if there's RLE or non-RLE pending
				If repeating Then
					'Repeat one more sample of RLE block
					For j As Byte = 0 To 3
						imagestart[wp + j] = sample(j)
					Next j
					wp += 4
					column += 1
					repeating -= 1
				ElseIf copying Then
					'Copy one more sample from buffer
					For j As Byte = 0 To 2
						imagestart[wp + j] = buffer[rp + j]
					Next j
					imagestart[wp + 3] = 255
					wp += 4 : rp += 3
					column += 1
					copying -= 1
				Else
					'Process an RLE/non-RLE header
					copying = buffer[rp]
					rp += 1
					If copying And 128 Then
						repeating = copying - 127
						copying = 0
						For j As Byte = 0 To 2
							sample(j) = buffer[rp + j]
						Next j
						sample(3) = 255
						rp += 3
					Else
						copying += 1
					End If
				End If
			Loop Until column = h.ImageWidth
		Next i
	'True colour, uncompressed, with alpha channel
	ElseIf h.PixelDepth = 32 And h.ImageType = 2 And alphachannel <> 0 Then
		rp = 0
		
		'For every line
		For i As Long = 0 To h.ImageHeight - 1
			'Calculate relative start of the line
			If inverted Then
				wp = (h.ImageHeight - i - 1) * linelength
			Else
				wp = i * linelength
			End If
			
			For j As Long = 0 To 4 * h.ImageWidth - 1
				imagestart[wp + j] = buffer[rp + j]
			Next j
			rp += 4 * h.ImageWidth
		Next i
	'True colour, uncompressed, without alpha channel
	ElseIf h.PixelDepth = 24 And h.ImageType = 2 And alphachannel = 0 Then
		rp = 0
		
		'For every line
		For i As Long = 0 To h.ImageHeight - 1
			'Calculate relative start of the line
			If inverted Then
				wp = (h.ImageHeight - i - 1) * linelength
			Else
				wp = i * linelength
			End If
			
			For j As Long = 0 To h.ImageWidth - 1
				imagestart[wp + 4 * j] = buffer[rp + 3 * j]
				imagestart[wp + 4 * j + 1] = buffer[rp + 3 * j + 1]
				imagestart[wp + 4 * j + 2] = buffer[rp + 3 * j + 2]
				imagestart[wp + 4 * j + 3] = 255
			Next j
			rp += 3 * h.ImageWidth
		Next i
	'High colour (16 bit), RLE compressed
	ElseIf h.PixelDepth = 16 And h.ImageType = 10 Then
		Dim As UByte copying, repeating
		Dim column As Long, sample As UShort
		
		rp = 0	'Reset read pointer
		
		'For every line
		For i As Long = 0 To h.ImageHeight - 1
			'Calculate relative start of the line
			If inverted Then
				wp = (h.ImageHeight - i - 1) * linelength
			Else
				wp = i * linelength
			End If
			
			'Starting the line from column zero
			column = 0
			Do
				'Check to see if there's RLE or non-RLE pending
				If repeating Then
					'Repeat one more sample of RLE block
					imagestart[wp] = sample And &hFF
					imagestart[wp + 1] = sample ShR 8
					wp += 2
					column += 1
					repeating -= 1
				ElseIf copying Then
					'Copy one more sample from buffer
					sample = buffer[rp] + 256 * buffer[rp + 1]
					sample = c15to16(sample)
					imagestart[wp] = sample And 255
					imagestart[wp + 1] = sample ShR 8
					wp += 2 : rp += 2
					column += 1
					copying -= 1
				Else
					'Process an RLE/non-RLE header
					copying = buffer[rp]
					rp += 1
					If copying And 128 Then
						repeating = copying - 127
						copying = 0
						sample = buffer[rp] + 256 * buffer[rp + 1]
						sample = c15to16(sample)
						rp += 2
					Else
						copying += 1
					End If
				End If
			Loop Until column = h.ImageWidth
		Next i
	'High colour (16 bit), uncompressed
	ElseIf h.PixelDepth = 16 And h.ImageType = 2 Then
		rp = 0
		
		'For every line
		For i As Long = 0 To h.ImageHeight - 1
			'Calculate relative start of the line
			If inverted Then
				wp = (h.ImageHeight - i - 1) * linelength
			Else
				wp = i * linelength
			End If
			
			For j As Long = 0 To h.ImageWidth - 1
				imagestart[wp + 2 * j] = buffer[rp + 2 * j]
				imagestart[wp + 2 * j + 1] = buffer[rp + 2 * j + 1]
			Next j
			rp += 2 * h.ImageWidth
		Next i
	'Indexed (8 bit), RLE compressed
	ElseIf h.PixelDepth = 8 And h.ImageType = 9 And (h.ColorMapDepth = 24 Or h.ColorMapDepth = 32) Then
		Dim As UByte copying, repeating
		Dim column As Long, sample As UByte
		
		rp = 0	'Reset read pointer
		
		'For every line
		For i As Long = 0 To h.ImageHeight - 1
			'Calculate relative start of the line
			If inverted Then
				wp = (h.ImageHeight - i - 1) * linelength
			Else
				wp = i * linelength
			End If
			
			'Starting the line from column zero
			column = 0
			Do
				'Check to see if there's RLE or non-RLE pending
				If repeating Then
					'Repeat one more sample of RLE block
					imagestart[wp] = sample
					wp += 1
					column += 1
					repeating -= 1
				ElseIf copying Then
					'Copy one more sample from buffer
					imagestart[wp] = buffer[rp]
					wp += 1 : rp += 1
					column += 1
					copying -= 1
				Else
					'Process an RLE/non-RLE header
					copying = buffer[rp]
					rp += 1
					If copying And 128 Then
						repeating = copying - 127
						copying = 0
						sample = buffer[rp]
						rp += 1
					Else
						copying += 1
					End If
				End If
			Loop Until column = h.ImageWidth
		Next i
		
		Dim spal As String
		
		spal = Space(h.ColorMapDepth * h.ColorMapLength)
		Get #f, 19 + h.IDlength, spal
		
		wp = 3 * h.ColorMapStart
		For i As Short = h.ColorMapStart To h.ColorMapStart + h.ColorMapLength - 1
			If pal = 0 Then
				Palette i, ASC(Mid(spal, 3, 1)), ASC(Mid(spal, 2, 1)), ASC(Left(spal, 1))
			Else
				pal[wp] = ASC(Left(spal, 1))
				pal[wp + 1] = ASC(Mid(spal, 2, 1))
				pal[wp + 2] = ASC(Mid(spal, 3, 1))
				wp += 3
			End If
			spal = Mid(spal, 1 + h.ColorMapDepth \ 8)
		Next i
	Else
		'Unsupported format
		TargaError = 205
		TargaErrorMessage = "Unsupported format"
		Deallocate buffer
		ImageDestroy iptr
		Close f
		Return 0
	End If
	
	Deallocate buffer
	Close f
	
	Return iptr
End Function


Sub TargaSave8(filename As String, image As Any Ptr, pal As String)
	Dim As Integer iwidth, iheight, bypp, linelength
	Dim As UByte Ptr imagestart
	Dim As String tpalette
		
	'Reset error information
	TargaError = 0 : TargaErrorMessage = ""
	
	'Make sure it's a valid image
	If image = 0 Then
		TargaError = 101
		TargaErrorMessage = "No image in buffer"
		Exit Sub
	End If
	ImageInfo image, iwidth, iheight, bypp, linelength, imagestart
	If bypp <> 1 Then
		TargaError = 102
		TargaErrorMessage = "Not a 8bit image. Unsupported"
		Exit Sub
	End If
	
	'Targa supports palettes of more than 256 colours and indeces
	'wider than one byte. It also supports an alpha attribute in the
	'color map. These will not be used.
		
	'Set up image header
	Dim h As TargaHeader, f As Short
	

	h.PixelDepth = 8
	h.ImageDescriptor = 0
	
	h.ColorMapType = 1
	h.ColorMapStart = 0
	h.ColorMapDepth = 24
	If Len(pal) Then
		h.ColorMapLength = Len(pal) \ 3
		tpalette = pal
	Else
		Dim As Integer r, g, b
		h.ColorMapLength = 256
		
		For i As Integer = 0 To 255
			Palette Get i, r, g, b
			tpalette &= Chr(b, g, r)
		Next i
	End If

	h.ImageType = 9
	h.ImageWidth = iwidth
	h.ImageHeight = iheight

	'Open file
	f = FreeFile
	If Open(filename For Output As f) Then
		TargaError = 103
		TargaErrorMessage = "Failed to create image file"
		Exit Sub
	Else
		Close f
		Open filename For Binary Access Write As f
	End If
	
	'Put header
	Put #f, 1, h
	
	'... and palette
	Put #f, , tpalette
	
	'Compress image row by row
	Dim rp As Long, column As Long, buffer As String
	Dim count As Short, status As Byte, sample As UByte
		
	For i As Long = 0 To iheight - 1	'For every row...
		'Calculate where to read the row from
		rp = (iheight - i - 1) * linelength
		
		buffer = ""
		column = 0
		status = 0	'Still don't know if RLE or not
		count = 0	'Nothing pending
		Do
			Select Case status
				Case 0 'Undefined
					sample = imagestart[rp + column]
					
					'If it's the last pixel, just push it
					If column = iwidth - 1 Then
						buffer &= Chr(0) + Chr(sample)
						Exit Do
					End If
					
					count = 0
					If sample = imagestart[rp + column + 1] Then
						status = 1	'Building an RLE block
					Else
						status = 2	'Building a non-RLE block
					End If
				Case 1	'RLE
					If imagestart[rp + column] = sample Then
						If count = 128 Then	'Block full. Push into the buffer
							buffer &= Chr(255) + Chr(sample)
							count = 0
							status = 0
						Else
							count += 1
							column += 1
						End If
					Else
						'Found an end for the RLE block
						buffer &= Chr(127 + count)
						buffer &= Chr(sample)
						count = 0
						status = 0
					End If
				Case Else	'Non-RLE
					If imagestart[rp + column] = imagestart[rp + column + 1]  Then
						'End of non-RLE block
						buffer &= Chr(count - 1)
						For j As Short = column - count To column - 1
							buffer &= Chr(imagestart[rp + j])
						Next j
						count = 0
						status = 0
					Else
						If count = 128 Then	'Block full. Push into the buffer
							buffer &= Chr(127)
							For j As Short = column - 128 To column - 1
								buffer &= Chr(imagestart[rp + j])
							Next j
							count = 0
							status = 0
						Else
							count += 1
							column += 1
						End If
					End If
			End Select
		Loop Until column = iwidth
		
		If column = iwidth Then
			If status = 1 Then
				buffer &= Chr(127 + count)
				buffer &= Chr(sample)
			Else
				buffer &= Chr(count - 1)
				For j As Short = column - count To column - 1
					buffer &= Chr(imagestart[rp + j])
				Next j
			End If
		End If
		
		Put #f, , buffer
	Next i
	
	'Targa v2.0 with no extensions
	buffer = String(8, 0) + "TRUEVISION-XFILE." + Chr(0)
	Put #f, , buffer
	
	Close f
End Sub


Sub TargaSave16(filename As String, image As Any Ptr)
	Dim As Integer iwidth, iheight, bypp, linelength
	Dim As UShort Ptr imagestart
		
	'Reset error information
	TargaError = 0 : TargaErrorMessage = ""
	
	'Make sure it's a valid image
	If image = 0 Then
		TargaError = 101
		TargaErrorMessage = "No image in buffer"
		Exit Sub
	End If
	ImageInfo image, iwidth, iheight, bypp, linelength, imagestart
	If bypp <> 2 Then
		TargaError = 102
		TargaErrorMessage = "Not a 16bit image. Unsupported"
		Exit Sub
	End If
	
	'In 16 bit mode, FreeBasic uses no alpha channel, but
	'Targa always uses 1 bit for alpha, so it will have to
	'be skipped and the green component stored as 5 bits.
		
	'Set up image header
	Dim h As TargaHeader, f As Short
	

	h.PixelDepth = 16
	h.ImageDescriptor = 0
	
	h.ImageType = 10
	h.ImageWidth = iwidth
	h.ImageHeight = iheight

	'Open file
	f = FreeFile
	If Open(filename For Output As f) Then
		TargaError = 103
		TargaErrorMessage = "Failed to create image file"
		Exit Sub
	Else
		Close f
		Open filename For Binary Access Write As f
	End If
	
	'Put header
	Put #f, 1, h
	
	'Compress image row by row
	Dim rp As Long, column As Long, buffer As String
	Dim count As Short, status As Byte, sample As UShort
	
	For i As Long = 0 To iheight - 1	'For every row...
		'Calculate where to read the row from
		rp = (iheight - i - 1) * linelength \ 2
		
		buffer = ""
		column = 0
		status = 0	'Still don't know if RLE or not
		count = 0	'Nothing pending
		Do
			Select Case status
				Case 0 'Undefined
					sample = imagestart[rp + column]
					
					'If it's the last pixel, just push it
					If column = iwidth - 1 Then
						buffer &= Chr(0) + MkShort(c16to15(sample))
						Exit Do
					End If
					
					count = 0
					If sample = imagestart[rp + column + 1] Then
						status = 1	'Building an RLE block
					Else
						status = 2	'Building a non-RLE block
					End If
				Case 1	'RLE
					If imagestart[rp + column] = sample Then
						If count = 128 Then	'Block full. Push into the buffer
							buffer &= Chr(255) + MkShort(c16to15(sample))
							count = 0
							status = 0
						Else
							count += 1
							column += 1
						End If
					Else
						'Found an end for the RLE block
						buffer &= Chr(127 + count)
						buffer &= MkShort(c16to15(sample))
						count = 0
						status = 0
					End If
				Case Else	'Non-RLE
					If imagestart[rp + column] = imagestart[rp + column + 1]  Then
						'End of non-RLE block
						buffer &= Chr(count - 1)
						For j As Short = column - count To column - 1
							buffer &= MkShort(c16to15(imagestart[rp + j]))
						Next j
						count = 0
						status = 0
					Else
						If count = 128 Then	'Block full. Push into the buffer
							buffer &= Chr(127)
							For j As Short = column - 128 To column - 1
								buffer &= MkShort(c16to15(imagestart[rp + j]))
							Next j
							count = 0
							status = 0
						Else
							count += 1
							column += 1
						End If
					End If
			End Select
		Loop Until column = iwidth
		
		If column = iwidth Then
			If status = 1 Then
				buffer &= Chr(127 + count)
				buffer &= MkShort(c16to15(sample))
			Else
				buffer &= Chr(count - 1)
				For j As Short = column - count To column - 1
					buffer &= MkShort(c16to15(imagestart[rp + j]))
				Next j
			End If
		End If
		
		Put #f, , buffer
	Next i
	
	'Targa v2.0 with no extensions
	buffer = String(8, 0) + "TRUEVISION-XFILE." + Chr(0)
	Put #f, , buffer
	
	Close f
End Sub


Sub TargaSave32(filename As String, image As Any Ptr)
	Dim As Integer iwidth, iheight, bypp, linelength
	Dim As ULong Ptr imagestart
		
	'Reset error information
	TargaError = 0 : TargaErrorMessage = ""
	
	'Make sure it's a valid image
	If image = 0 Then
		TargaError = 101
		TargaErrorMessage = "No image in buffer"
		Exit Sub
	End If
	ImageInfo image, iwidth, iheight, bypp, linelength, imagestart
	If bypp <> 4 Then
		TargaError = 102
		TargaErrorMessage = "Not a 32bit image. Unsupported"
		Exit Sub
	End If
	
	'See if the image contains any alpha information
	Dim alphachannel As Byte = 0
	For i As Long = 0 To iwidth * iheight - 1
		If imagestart[i] ShR 24 <> 255 Then
			alphachannel = -1
			Exit For
		End If
	Next i
	
	'Set up image header
	Dim h As TargaHeader, f As Short
	
	If alphachannel Then
		h.PixelDepth = 32
		h.ImageDescriptor = 8
	Else
		h.PixelDepth = 24
		h.ImageDescriptor = 0
	End If
	
	h.ImageType = 10
	h.ImageWidth = iwidth
	h.ImageHeight = iheight

	'Open file
	f = FreeFile
	If Open(filename For Output As f) Then
		TargaError = 103
		TargaErrorMessage = "Failed to create image file"
		Exit Sub
	Else
		Close f
		Open filename For Binary Access Write As f
	End If
	
	'Put header
	Put #f, 1, h
	
	'Compress image row by row
	Dim rp As Long, column As Long, buffer As String
	Dim count As Short, status As Byte, sample As ULong
	
	For i As Long = 0 To iheight - 1	'For every row...
		'Calculate where to read the row from
		rp = (iheight - i - 1) * linelength \ 4
		
		buffer = ""
		column = 0
		status = 0	'Still don't know if RLE or not
		count = 0	'Nothing pending
		Do
			Select Case status
				Case 0 'Undefined
					sample = imagestart[rp + column]
					
					'If it's the last pixel, just push it
					If column = iwidth - 1 Then
						If alphachannel Then
							buffer &= Chr(0) + MkL(sample)
						Else
							buffer &= Chr(0) + Left(MkL(sample), 3)
						End If
						Exit Do
					End If
					
					count = 0
					If sample = imagestart[rp + column + 1] Then
						status = 1	'Building an RLE block
					Else
						status = 2	'Building a non-RLE block
					End If
				Case 1	'RLE
					If imagestart[rp + column] = sample Then
						If count = 128 Then	'Block full. Push into the buffer
							If alphachannel Then
								buffer &= Chr(255) + MkL(sample)
							Else
								buffer &= Chr(255) + Left(MkL(sample), 3)
							End If
							count = 0
							status = 0
						Else
							count += 1
							column += 1
						End If
					Else
						'Found an end for the RLE block
						buffer &= Chr(127 + count)
						If alphachannel Then
							buffer &= MkL(sample)
						Else
							buffer &= Left(MkL(sample), 3)
						End If
						count = 0
						status = 0
					End If
				Case Else	'Non-RLE
					If imagestart[rp + column] = imagestart[rp + column + 1]  Then
						'End of non-RLE block
						buffer &= Chr(count - 1)
						For j As Short = column - count To column - 1
							If alphachannel Then
								buffer &= MkL(imagestart[rp + j])
							Else
								buffer &= Left(MkL(imagestart[rp + j]), 3)
							End If
						Next j
						count = 0
						status = 0
					Else
						If count = 128 Then	'Block full. Push into the buffer
							buffer &= Chr(127)
							For j As Short = column - 128 To column - 1
								If alphachannel Then
									buffer &= MkL(imagestart[rp + j])
								Else
									buffer &= Left(MkL(imagestart[rp + j]), 3)
								End If
							Next j
							count = 0
							status = 0
						Else
							count += 1
							column += 1
						End If
					End If
			End Select
		Loop Until column = iwidth
		
		If column = iwidth Then
			If status = 1 Then
				buffer &= Chr(127 + count)
				If alphachannel Then
					buffer &= MkL(sample)
				Else
					buffer &= Left(MkL(sample), 3)
				End If
			Else
				buffer &= Chr(count - 1)
				For j As Short = column - count To column - 1
					If alphachannel Then
						buffer &= MkL(imagestart[rp + j])
					Else
						buffer &= Left(MkL(imagestart[rp + j]), 3)
					End If
				Next j
			End If
		End If
		
		Put #f, , buffer
	Next i
	
	'Targa v2.0 with no extensions
	buffer = String(8, 0) + "TRUEVISION-XFILE." + Chr(0)
	Put #f, , buffer
	
	Close f
End Sub


Sub TargaSave(filename As String, image As Any Ptr)
	TargaSave32 filename, image
End Sub
